# encoding: UTF-8

import logging
import re

from ws_properties.utils.logs import get_logger_for_instance

from dns_hosting.services.support.stream import StreamReader


class NSDLogger(object):
    LOG_ENTRY_PATTERN = re.compile(
        r'^\[(\d+)\] (\w+)\[(\d+)\]: (\w+): (.*)$'
    )

    def __init__(self, stream, logger=None):
        self.__reader = StreamReader(stream)
        self.__logger = logger or get_logger_for_instance(self)

    def start(self):
        self.__reader.readline(self.__line_read)

    def __line_read(self, _, error, line):
        if line:
            self.__handle_line(line)

            if not error:
                self.__reader.readline(self.__line_read)

    def __handle_line(self, line):
        line = line.strip()
        if line:
            match = self.LOG_ENTRY_PATTERN.match(line)
            if match:
                try:
                    timestamp, pname, pid, level, message = map(str, match.groups())
                    level = self.__normalize_level(level)

                    if self.__logger.isEnabledFor(level):
                        timestamp = int(timestamp)

                        record = self.__logger.makeRecord(
                            name=self.__logger.name,
                            level=level,
                            fn='nsd',
                            lno=0,
                            msg=message,
                            args=[],
                            exc_info=None,
                        )
                        record.created = timestamp
                        record.msecs = 0
                        record.process = pid
                        record.processName = pname
                        record.relativeCreated = (timestamp - logging._startTime) * 1000

                        self.__logger.handle(record)
                except Exception:
                    self.__logger.warning(
                        'Failed to parse line: %s',
                        line,
                        exc_info=1,
                    )
            else:
                self.__logger.warning('Failed to parse line: %s', line)

    def __normalize_level(self, name):
        if name == 'notice':
            name = 'info'
        return logging.getLevelName(name.upper())
