# encoding: UTF-8

import functools
import logging
import subprocess

from ws_properties.utils.logs import get_logger_for_instance

from dns_hosting.services.dnsserver.nsd.logger import NSDLogger
from dns_hosting.services.support.process import PopenSupervisor
from dns_hosting.services.support.retry import StandardRetryPolicy


class NSDSupervisor(object):
    def __init__(
            self,
            config_file,
            executable='nsd',
            graceful_shutdown_timeout=60,
            restart_policy=StandardRetryPolicy(max_attempts=None),
    ):
        self.__supervisor = PopenSupervisor(
            args=[
                executable,
                '-d',
                '-c', config_file,
            ],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            graceful_shutdown_timeout=graceful_shutdown_timeout,
            restart_policy=restart_policy,
        )
        self._logger = get_logger_for_instance(self)
        self._proc_logger = get_logger_for_instance(self)
        self._proc_logger.setLevel(logging.WARNING)

    def start(self, start_cb, exit_cb):
        self.__supervisor.start(
            functools.partial(
                self.__nsd_started,
                start_cb,
            ),
            functools.partial(
                self.__nsd_exited,
                exit_cb,
            ),
        )

    def stop(self):
        self.__supervisor.stop()

    def restart(self):
        self.__supervisor.process.terminate()

    def __nsd_started(self, start_cb, _, error):
        if not error:
            self._logger.info('NSD process started.')
            NSDLogger(self.__supervisor.process.stdout, self._proc_logger).start()
            NSDLogger(self.__supervisor.process.stderr, self._proc_logger).start()

        start_cb(self, error)

    def __nsd_exited(self, exit_cb, _, exitcode):
        self._logger.info('NSD process exited: %d', exitcode)
        exit_cb(self, exitcode)
