# encoding: UTF-8

import functools

import gevent
import ijson
import requests
from ws_properties.utils.logs import get_logger_for_instance

from dns_hosting.services.support.retry import StandardRetryPolicy


class ZonesTracker(object):
    def __init__(
            self,
            host,
            port,
            sync_interval=900,
            retry_policy=StandardRetryPolicy(),
    ):
        self.host = host
        self.port = port
        self.sync_interval = sync_interval
        self.retry_policy = retry_policy

        self.__running = False
        self.__start_event = None

        self._logger = get_logger_for_instance(self)

    def __del__(self):
        self.stop()

    @property
    def running(self):
        return self.__running

    def start(self, zone_names_change_cb):
        if self.__running:
            raise AssertionError('ZonesTracker already started.')
        else:
            self.__running = True
            self.__start_event = None

            delay_sequence = self.retry_policy.delay_sequence()
            self._upload_zone_names(zone_names_change_cb, delay_sequence)

    def stop(self):
        if self.__running:
            if self.__start_event:
                self.__cancel_start()
                self.__running = False
                return True
            else:
                self.__running = False
                return False
        else:
            return True

    def __cancel_start(self):
        self.__start_event.stop()
        self.__start_event.close()
        self.__start_event = None

    def _make_url(self):
        if ':' in self.host:
            return 'http://[%s]:%d/' % (self.host, self.port)
        else:
            return 'http://%s:%d/' % (self.host, self.port)

    def _upload_zone_names(self, zone_names_change_cb, delay_sequence):
        if self.__running:
            try:
                _, delay = next(delay_sequence)
            except StopIteration:
                pass
            else:
                url = self._make_url()
                g = gevent.Greenlet(requests.get, url, stream=True)
                g.link(
                    functools.partial(
                        self._zone_names_upload,
                        zone_names_change_cb,
                        delay_sequence,
                    )
                )

                if delay:
                    self.__start_event = g.parent.loop.timer(delay)
                    self.__start_event.start(g.start)
                    self._logger.info(
                        'Zone upload scheduled within %.3f seconds.',
                        delay,
                    )
                else:
                    self.__start_event = g.parent.loop.run_callback(g.start)
                    self._logger.info('Zone upload scheduled now.')

    def _zone_names_upload(self, zone_names_change_cb, delay_sequence, result):
        self.__cancel_start()

        try:
            response = result.get()
        except Exception as e:
            self._logger.exception('Zone upload failed.')
            self._upload_zone_names(zone_names_change_cb, delay_sequence)
            self._run_callback(zone_names_change_cb, e, None)
        else:
            try:
                response.raise_for_status()

                self._logger.info('Zone upload started.')

                zone_names = [
                    value
                    for prefix, event, value in ijson.parse(response.raw)
                    if event == 'string'
                ]
            except Exception as e:
                self._logger.exception('Zone upload failed.')
                self._upload_zone_names(zone_names_change_cb, delay_sequence)
                self._run_callback(zone_names_change_cb, e, None)
            else:
                self._logger.info('Zone upload completed.')
                delay_sequence = self.retry_policy.delay_sequence(
                    initial_delay=self.sync_interval,
                )
                self._upload_zone_names(zone_names_change_cb, delay_sequence)
                self._run_callback(zone_names_change_cb, None, zone_names)
            finally:
                response.close()

    def _run_callback(self, zone_names_change_cb, error, zone_names):
        g = gevent.getcurrent()
        g.parent.loop.run_callback(
            zone_names_change_cb,
            self,
            error,
            zone_names,
        )
