# encoding: UTF-8

import functools
import signal

import gevent.event
import gevent.lock
import gevent.subprocess

from dns_hosting.services.support.retry import StandardRetryPolicy


class Popen(object):
    def __init__(self, args, bufsize=0, executable=None,
                 stdin=None, stdout=None, stderr=None,
                 preexec_fn=None, close_fds=False, shell=False,
                 cwd=None, env=None, universal_newlines=False,
                 startupinfo=None, creationflags=0):
        self.args = args
        self.bufsize = bufsize
        self.executable = executable
        self.stdin = stdin
        self.stdout = stdout
        self.stderr = stderr
        self.preecec_fn = preexec_fn
        self.close_fds = close_fds
        self.shell = shell
        self.cwd = cwd
        self.env = env
        self.universal_newlines = universal_newlines
        self.startupinfo = startupinfo
        self.creationflags = creationflags

        self.__running = False
        self.__start_event = None
        self.__scheduled_stop = None
        self.__process = None  # type: gevent.subprocess.Popen

    def __del__(self):
        self.stop()

    @property
    def running(self):
        return self.__running

    @property
    def process(self):
        return self.__process

    def start(self, start_cb, exit_cb, delay=None):
        if self.__running:
            raise AssertionError('Process already started.')

        self.__running = True
        self.__scheduled_stop = None
        self.__process = None

        g = gevent.Greenlet(self.__popen)
        g.rawlink(
            functools.partial(
                self.__process_started,
                start_cb,
                exit_cb,
            ),
        )

        if delay:
            self.__start_event = g.parent.loop.timer(delay)
            self.__start_event.start(g.start)
        else:
            self.__start_event = g.parent.loop.run_callback(g.start)

    def stop(self, sig=signal.SIGTERM):
        if self.__running:
            if self.__process:
                self.__process.send_signal(sig)
                return False
            elif self.__start_event:
                self.__cancel_start()
                self.__running = False
                return True
            else:
                self.__scheduled_stop = sig
                return False
        else:
            return True

    def __cancel_start(self):
        self.__start_event.stop()
        self.__start_event.close()
        self.__start_event = None

    def __popen(self):
        self.__cancel_start()

        return gevent.subprocess.Popen(
            args=self.args,
            bufsize=self.bufsize,
            executable=self.executable,
            stdin=self.stdin,
            stdout=self.stdout,
            stderr=self.stderr,
            preexec_fn=self.preecec_fn,
            close_fds=self.close_fds,
            shell=self.shell,
            cwd=self.cwd,
            env=self.env,
            universal_newlines=self.universal_newlines,
            startupinfo=self.startupinfo,
            creationflags=self.creationflags,
        )

    def __process_started(self, start_cb, exit_cb, result):
        try:
            self.__process = result.get()
        except Exception as e:
            self.__running = False
            start_cb(self, e)
        else:
            self.__process.rawlink(
                functools.partial(
                    self.__process_exited,
                    exit_cb,
                )
            )

            if self.__scheduled_stop is not None:
                self.__process.send_signal(self.__scheduled_stop)

            start_cb(self, None)

    def __process_exited(self, exit_cb, popen):
        self.__running = False
        exit_cb(self, popen.returncode)


class PopenSupervisor(Popen):
    KILL_SIGNAL = signal.SIGKILL

    def __init__(self, args, bufsize=0, executable=None,
                 stdin=None, stdout=None, stderr=None,
                 preexec_fn=None, close_fds=False, shell=False,
                 cwd=None, env=None, universal_newlines=False,
                 startupinfo=None, creationflags=0,
                 termination_signal=signal.SIGTERM,
                 graceful_shutdown_timeout=60,
                 restart_policy=StandardRetryPolicy()):
        super(PopenSupervisor, self).__init__(
            args, bufsize, executable, stdin, stdout, stderr, preexec_fn,
            close_fds, shell, cwd, env, universal_newlines, startupinfo,
            creationflags,
        )
        self.termination_signal = termination_signal
        self.graceful_shutdown_timeout = graceful_shutdown_timeout
        self.restart_policy = restart_policy

        self.__running = False
        self.__scheduled_kill = None  # type: gevent.Greenlet

    def start(self, start_cb, exit_cb):
        if self.__running:
            raise AssertionError('Supervisor already started.')

        self.__running = True

        delay_sequence = self.restart_policy.delay_sequence()
        try:
            _, delay = next(delay_sequence)
        except StopIteration:
            pass
        else:
            self.__start_process(delay, start_cb, exit_cb, delay_sequence)

    def stop(self):
        self.__running = False
        if super(PopenSupervisor, self).stop(self.termination_signal):
            return True
        else:
            self.__scheduled_kill = gevent.spawn_later(
                self.graceful_shutdown_timeout,
                super(PopenSupervisor, self).stop,
                self.KILL_SIGNAL,
            )
            return False

    def __start_process(self, delay, start_cb, exit_cb, delay_sequence):
        if self.__running:
            super(PopenSupervisor, self).start(
                functools.partial(
                    self.__process_started,
                    start_cb,
                ),
                functools.partial(
                    self.__process_exited,
                    start_cb,
                    exit_cb,
                    delay_sequence,
                ),
                delay,
            )

    def __process_started(self, start_cb, _, error):
        if error:
            self.__running = False

        start_cb(self, error)

    def __process_exited(self, start_cb, exit_cb, delay_sequence, _, exitcode):
        if self.__scheduled_kill:
            self.__scheduled_kill.kill(block=False)

        try:
            _, delay = next(delay_sequence)
        except StopIteration:
            self.__running = False
            exit_cb(self, exitcode)
        else:
            try:
                exit_cb(self, exitcode)
            finally:
                self.__start_process(delay, start_cb, exit_cb, delay_sequence)


