# encoding: UTF-8

import itertools

import gevent

_missing = object()


class _AttemptContext(object):
    def __init__(self, policy, number, delay):
        self.policy = policy
        self.number = number
        self.delay = delay
        self.__exc_info = None

    @property
    def success(self):
        return self.__exc_info is None

    @property
    def exc_info(self):
        return self.__exc_info

    def __enter__(self):
        return self.number

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type:
            self.__exc_info = (exc_type, exc_val, exc_tb)
            if self.policy.can_retry(exc_type, exc_val, exc_tb):
                return True


def retry(policy, auto_delay=True):
    ctx = None

    for attempt, delay in policy.delay_sequence():
        if auto_delay:
            gevent.sleep(delay)

        ctx = _AttemptContext(policy, attempt, delay)
        yield ctx

        if ctx.success:
            raise StopIteration

    if not ctx.success:
        raise ctx.exc_info[0], ctx.exc_info[1], ctx.exc_info[2]



class RetryPolicy(object):
    def can_retry(self, exc_type, exc_val, tb):
        raise NotImplementedError

    def delay_sequence(self, *args, **kwargs):
        raise NotImplementedError

    def first_delay(self, *args, **kwargs):
        try:
            return next(self.delay_sequence(*args, **kwargs))[1]
        except StopIteration:
            return None


class StandardRetryPolicy(RetryPolicy):
    def __init__(self, delay=5, max_attempts=3, initial_delay=0):
        self.delay = delay
        self.max_attempts = max_attempts
        self.initial_delay = initial_delay

    def can_retry(self, exc_type, exc_val, tb):
        return isinstance(exc_val, Exception)

    def delay_sequence(
            self,
            delay=_missing,
            max_attempts=_missing,
            initial_delay=_missing,
    ):
        if delay is _missing:
            delay = self.delay

        if max_attempts is _missing:
            max_attempts = self.max_attempts

        if initial_delay is _missing:
            initial_delay = self.initial_delay

        if max_attempts is None:
            attempts_it = itertools.count()
        else:
            attempts_it = xrange(max_attempts)

        for attempt in attempts_it:
            if attempt == 0:
                yield attempt + 1, initial_delay
            else:
                yield attempt + 1, delay
