import time
import logging

import six
import json


class DeployFormatter(logging.Formatter):
    COMMON_FIELDS = (
        ('levelname', 'levelStr'),
        ('levelno', 'level'),
        ('asctime', '@timestamp'),
        ('name', 'loggerName'),
        ('threadName', 'threadName'),
        ('message', 'message'),
    )

    COMMON_CONTEXT_FIELDS = (
        ('process', 'pid'),
        ('processName', 'process'),
        ('thread', 'thread_id'),
        ('funcName', 'func_name'),
        ('lineno', 'lineno'),
        ('module', 'module'),
        ('pathname', 'filename'),
    )

    KNOWN_KEYS = {
        'name', 'msg', 'message', 'args', 'levelname',
        'levelno', 'pathname', 'filename', 'module',
        'exc_info', 'exc_text', 'lineno', 'funcName',
        'created', 'msecs', 'relativeCreated', 'thread',
        'threadName', 'processName', 'process', 'asctime',
    }

    def formatTime(self, record, datefmt=None):
        gt = time.gmtime(record.created)
        return time.strftime('%Y-%m-%dT%H:%M:%S.%%03dZ', gt) % (record.msecs,)

    def format(self, record):
        record.message = record.getMessage()
        record.asctime = self.formatTime(record)

        msg = {}
        for field, dest_name in self.COMMON_FIELDS:
            if hasattr(record, field):
                msg[dest_name] = getattr(record, field)

        msg['@fields'] = {}
        for field, dest_name in self.COMMON_CONTEXT_FIELDS:
            if hasattr(record, field):
                msg['@fields'][dest_name] = getattr(record, field)

        for k, v in six.iteritems(record.__dict__):
            if k not in self.KNOWN_KEYS:
                msg['@fields'][k] = v

        if record.exc_info and not record.exc_text:
            record.exc_text = self.formatException(record.exc_info)
        if record.exc_text:
            msg['stackTrace'] = record.exc_text

        return json.dumps(msg)

class TskvFormatter(logging.Formatter):
    FIELDS = [
        ('levelname', 'levelStr'),
        ('asctime', '@timestamp'),
        ('name', 'loggerName'),
        ('funcName', 'func_name'),
        ('pathname', 'filename'),
        ('module', 'module'),
        ('process', 'pid'),
        ('message', 'message'),
    ]

    TSKV_PREFIX = 'tskv'
    
    def quote(self, value):
        return (str(value).
                replace('=', '\\=').
                encode('unicode_escape').
                decode('ascii'))

    def formatTime(self, record, datefmt=None):
        gt = time.gmtime(record.created)
        return time.strftime('%Y-%m-%dT%H:%M:%S.%%03dZ', gt) % (record.msecs,)

    def format(self, record):
        record.message = record.getMessage()
        record.asctime = self.formatTime(record)

        parts = [self.TSKV_PREFIX]
        for field, key in self.FIELDS:
            if hasattr(record, field):
                value = getattr(record, field)
                if key == 'filename':
                    value += ':' + str(getattr(record, 'lineno'))
                parts.append(self.quote(key) + '=' + self.quote(value))


        if record.exc_info and not record.exc_text:
            record.exc_text = self.formatException(record.exc_info)
        if record.exc_text:
            parts.append(self.quote('stackTrace') + '=' + self.quote(record.exc_text))

        return '\t'.join(parts)
