# encoding: UTF-8

import logging
import math
import time
import warnings

from ws_properties.utils.logs import get_logger_for_instance

_STANDARD_SCALE_MAP = {
    0: ('', 10. ** 0),
    1: ('k', 10. ** 3),
    2: ('M', 10. ** 6),
    3: ('G', 10. ** 9),
    4: ('T', 10. ** 12),
}


def standard_scale(value):
    scale = int(math.log10(abs(max(value, 1)))) // 3
    suffix, divisor = _STANDARD_SCALE_MAP.get(scale, 4)
    return '%.3f%s' % (value / divisor, suffix)


class ProgressLogger(object):
    def __init__(
            self,
            message=None,
            logger=None,
            interval=5,
            level=logging.INFO,
            scale=standard_scale,
    ):
        self.message = message
        self.logger = logger or get_logger_for_instance(self)
        self.interval = interval
        self.level = level
        self.scale = scale

        self._started = False
        self._value = 0
        self._prev = 0
        self._logged_at = 0

    @property
    def value(self):
        return self._value

    def start(self):
        self._started = True
        self._value = 0
        self._logged_at = time.time()

    def stop(self):
        self.log_progress()
        self._started = False

    def __enter__(self):
        self.start()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.stop()

    def update(self, n=1):
        if not self._started:
            warnings.warn(
                'ProgressLogger not started. All updates will be ignored.',
                UserWarning,
            )
            return

        self._value += n
        if time.time() - self._logged_at > self.interval:
            self.log_progress()

    def log_progress(self):
        self.logger.log(
            self.level,
            '%s%sit (%sit/s)',
            self.message + ': ' if self.message else '',
            self.scale(self._value),
            self.scale((self._value - self._prev) // (time.time() - self._logged_at)),
        )
        self._prev = self._value
        self._logged_at = time.time()


class IteratorLogger(ProgressLogger):
    def __init__(
            self,
            iterable,
            message=None,
            logger=None,
            interval=5,
            level=logging.INFO,
            scale=standard_scale,
    ):
        super(IteratorLogger, self).__init__(
            message,
            logger,
            interval,
            level,
            scale,
        )
        self.__iter = iter(iterable)

    def __iter__(self):
        return self

    def next(self):
        try:
            value = next(self.__iter)
        except StopIteration:
            self.log_progress()
            raise
        else:
            self.update()
            return value
