# encoding: UTF-8

import functools
import itertools
import time


class RetryPolicy(object):
    def apply(self, attempts, num, exc_type, exc_val, exc_tb):
        userdata = attempts.userdata
        common_args = (userdata, num, exc_type, exc_val, exc_tb)
        if self._is_attempt_successful(*common_args):
            attempts.complete()
        elif self._is_need_more_attempts(*common_args):
            delay = self._get_next_attempt_delay(*common_args)
            time.sleep(delay)
        else:
            raise exc_type, exc_val, exc_tb

    def __iter__(self):
        user_data = self._get_attempts_userdata()
        return Attempts(self, user_data)

    def __call__(self, *args, **kwargs):
        user_data = self._get_attempts_userdata(*args, **kwargs)
        return Attempts(self, user_data)

    def _get_attempts_userdata(self, *args, **kwargs):
        raise NotImplementedError

    def _is_attempt_successful(self, userdata, num, exc_type, exc_val, exc_tb):
        raise NotImplementedError

    def _is_need_more_attempts(self, userdata, num, exc_type, exc_val, exc_tb):
        raise NotImplementedError

    def _get_next_attempt_delay(self, userdata, num, exc_type, exc_val, exc_tb):
        raise NotImplementedError


class Attempts(object):
    def __init__(self, policy, userdata):
        self.__policy = policy
        self.__counter = itertools.count(1)
        self.__completed = False
        self.userdata = userdata

    def complete(self):
        self.__completed = True

    def __iter__(self):
        return self

    def next(self):
        if self.__completed:
            raise StopIteration
        else:
            return Attempt(
                self.__policy,
                self,
                next(self.__counter)
            )


class Attempt(object):
    def __init__(self, policy, attempts, number):
        self.__policy = policy  # type: RetryPolicy
        self.__attempts = attempts  # type: Attempts
        self.__number = number

    @property
    def number(self):
        return self.__number

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.__policy.apply(
            self.__attempts,
            self.__number,
            exc_type,
            exc_val,
            exc_tb,
        )
        return True


class StandardRetryPolicy(RetryPolicy):
    def __init__(
            self,
            delay=5,
            max_attempts=3,
    ):
        self.delay = delay
        self.max_attempts = max_attempts

    def _get_attempts_userdata(self, **options):
        return self._UserData(
            delay=options.get('delay', self.delay),
            max_attempts=options.get('max_attempts', self.max_attempts),
        )

    def _is_attempt_successful(self, userdata, num, exc_type, exc_val, exc_tb):
        return exc_type is None

    def _is_need_more_attempts(self, userdata, num, exc_type, exc_val, exc_tb):
        return num < userdata.max_attempts

    def _get_next_attempt_delay(self, userdata, num, exc_type, exc_val, exc_tb):
        return userdata.delay

    class _UserData(object):
        def __init__(self, delay, max_attempts):
            self.delay = delay
            self.max_attempts = max_attempts


def retry(policy):
    def wrapper(func):
        @functools.wraps(func)
        def wrapped(*args, **kwargs):
            for attempt in policy:
                with attempt:
                    return func(*args, **kwargs)

        return wrapped

    return wrapper
