# encoding: UTF-8

import flask.views
import threading
import retrying
import sqlalchemy.exc
import sqlalchemy.orm.exc
from appcore.tx.plugin import tx_manager
from ws_properties.utils.logs import get_logger_for_instance

from appcore.data.session import master_session
from appcore.injection import Injected
from appcore.injection import inject
from appcore.security.decorators import authenticated_client
from appcore.security.exceptions import UnauthorizedAccessError
from appcore.security.plugin import auth_identity
from appcore.web.decorators import parse_body
from appcore.web.decorators import punycode_domain_name
from appcore.web.decorators import render
from appcore.web.decorators import render_page
from appcore.web.decorators import supports_pagination
from appcore.web.helpers import abort
from dns_hosting.dao.domains import DomainRepository
from dns_hosting.dao.domains import RecordRepository
from dns_hosting.dto.domains import DomainSchema
from dns_hosting.dto.domains import RecordSchema
from dns_hosting.models.domains import Domain, Record, RecordType
from dns_hosting.services.auth.directory import DirectoryClient
from dns_hosting.services.auth.fouras import FourasClient, FourasNotFoundError,FourasError
from dns_hosting.services.auth.model import Scope


def is_tech(domain_name):
    return domain_name.endswith('.yaconnect.com.')


def retry_if_error(exception):
    return exception is FourasError


class DkimRecordInitThread(threading.Thread):

    master_session_factory = Injected('master_session_factory')  # type: engine.Engine
    fouras_client = Injected('fouras_client')  # type: FourasClient

    def __init__(self, domain_name):
        super(DkimRecordInitThread, self).__init__()
        self.domain_name = domain_name
        self.daemon = True
        self._logger = get_logger_for_instance(self)

    @retrying.retry(stop_max_attempt_number=5, wait_fixed=60000, retry_on_exception=retry_if_error)
    def run(self):
        from dns_hosting.app import application
        self._logger.info('Start init DKIM')
        with application.app_context(), tx_manager:
            domain_repository = DomainRepository(self.master_session_factory)
            record_repository = RecordRepository(self.master_session_factory)
            domain = domain_repository.find_by_name(self.domain_name)
            if not domain:
                return
            for txt_record in record_repository.find_all_by_domain_name_and_record_type(
                    self.domain_name,
                    RecordType.TXT
            ):
                if 'v=DKIM' in txt_record.content:
                    return
            try:
                public_key = self.fouras_client.domain_status(self.domain_name, 'mail')['public_key']
            except FourasNotFoundError:
                public_key = self.fouras_client.domain_key_gen(self.domain_name, 'mail')['public_key']

            rec = Record(
                name='mail._domainkey',
                type=RecordType.TXT,
                content='"{}"'.format(public_key),
                ttl=21600,
            )
            rec.domain = domain
            record_repository.save(rec, flush=False)
            self._logger.warning('DKIM record added')


class DomainMethodView(flask.views.MethodView):
    decorators = (
        authenticated_client(scopes=Scope.manage_domains),
    )

    domain_repository = Injected('domain_repository')  # type: DomainRepository
    record_repository = Injected('record_repository')  # type: RecordRepository
    directory_client = Injected('directory_client')  # type: DirectoryClient

    def __init__(self, *args, **kwargs):
        super(DomainMethodView, self).__init__(*args, **kwargs)
        self._logger = get_logger_for_instance(self)

    def _get_org_id(self):
        try:
            return flask.request.headers['X-Org-ID']
        except KeyError:
            raise UnauthorizedAccessError('Organization ID not provided')

    def _has_enough_permissions(self, org_id):
        if Scope.manage_any_domain in auth_identity.client_scopes:
            return True
        if not auth_identity.user_id:
            return False
        return self.directory_client.is_admin(org_id, auth_identity.user_id)

    def _list_domains(self):
        org_id = self._get_org_id()
        if self._has_enough_permissions(org_id):
            return self.directory_client.list_domains(
                org_id=org_id,
                include_technical=True,
            )
        else:
            return []

    def _require_permission(self, domain_name):
        can_manage_any_domain = (
            Scope.manage_any_domain in auth_identity.client_scopes
        )
        can_manage_technical = (
            Scope.manage_technical in auth_identity.client_scopes
        )
        if not can_manage_any_domain and domain_name not in self._list_domains():
            self._logger.warning('Unauthorized access attempt')
            abort(404, message='Domain not found', code='domain_not_found')
        if is_tech(domain_name) and not can_manage_technical:
            self._logger.warning('Access attempt to tech domain without scope')
            abort(403, message='No scope manage_technical', code='scope_required')

    def _disable_domain_pdd_sync(self, domain):
        if domain.pdd_sync_enabled:
            domain.pdd_sync_enabled = False
            domain.org_id = self._get_org_id()
            domain.revision = 1
            self.domain_repository.session.flush([domain])


class DomainCollectionView(DomainMethodView):
    MSG_DUPLICATES_DOMAIN = 'Duplicates another domain.'

    @supports_pagination
    @render_page(DomainSchema)
    def get(self, pageable):
        org_domains = self._list_domains()
        return self.domain_repository.find_paged(
            pageable=pageable,
            filter=(Domain.name.in_(org_domains))
        )

    @master_session
    @parse_body(DomainSchema, exclude=['id'])
    @render(DomainSchema, status_code=201)
    def post(self, body):
        self._require_permission(body.name)

        body.serial = 1
        body.pdd_sync_enabled = False
        body.org_id = self._get_org_id()
        body.revision = 1

        try:
            domain = self.domain_repository.save(body, flush=True)
            self._init_default_records(domain.name)
            return domain
        except sqlalchemy.exc.IntegrityError as e:
            self._handle_integrity_error(e)
            raise

    def _handle_integrity_error(self, e):
        duplication = (
                e.orig.pgcode == '23505' and
                e.orig.diag.constraint_name == 'idx_domain_name'
        )
        if duplication:
            abort(
                422,
                message='Invalid request body',
                errors=dict(_schema=[self.MSG_DUPLICATES_DOMAIN]),
                code='invalid_request_body',
            )

    def _init_default_records(self, domain_name):
        domain = self.domain_repository.find_by_name(domain_name)
        if not domain:
            return

        has_mx_record = bool(
            self.record_repository.find(domain_name, record_type=RecordType.MX)
        )
        has_cname_record = bool(
            self.record_repository.find(domain_name, record_type=RecordType.CNAME, name='mail')
        )

        has_spf_record = False
        for txt_record in self.record_repository.find_all_by_domain_name_and_record_type(
                domain_name,
                RecordType.TXT
        ):
            if 'spf' in txt_record.content:
                has_spf_record = True
                break

        if not has_mx_record:
            rec = Record(
                name='@',
                type=RecordType.MX,
                content='10 mx.yandex.net.',
                ttl=21600,
            )
            rec.domain = domain
            self.record_repository.save(rec, flush=False)
        if not has_spf_record:
            rec = Record(
                name='@',
                type=RecordType.TXT,
                content='"v=spf1 redirect=_spf.yandex.net"',
                ttl=21600,
            )
            rec.domain = domain
            self.record_repository.save(rec, flush=False)
        if not has_cname_record:
            rec = Record(
                name='mail',
                type=RecordType.CNAME,
                content='domain.mail.yandex.net.',
                ttl=21600,
            )
            rec.domain = domain
            self.record_repository.save(rec, flush=False)
        self.record_repository.session.flush()
        DkimRecordInitThread(domain_name).start()


class DomainDetailsView(DomainMethodView):
    @render(DomainSchema)
    @punycode_domain_name
    def get(self, domain_name):
        self._require_permission(domain_name)

        try:
            domain = self.domain_repository.find_by_name(domain_name)
        except sqlalchemy.orm.exc.NoResultFound:
            abort(404, message='Domain not found', code='domain_not_found')
        else:
            return domain

    @master_session
    @render(DomainSchema)
    @punycode_domain_name
    def delete(self, domain_name):
        self._require_permission(domain_name)

        try:
            domain = self.domain_repository.find_by_name(domain_name)
        except sqlalchemy.orm.exc.NoResultFound:
            abort(404, message='Domain not found', code='domain_not_found')
        else:
            self.domain_repository.delete(domain)
            return domain


def record_provider(record_id):
    try:
        domain_name = flask.request.view_args['domain_name'].encode('idna')
        return inject('record_repository').find_by_domain_name_and_id(
            domain_name=domain_name,
            record_id=record_id,
        )
    except sqlalchemy.orm.exc.NoResultFound:
        abort(404, message='Record not found', code='record_not_found')


class RecordCollectionView(DomainMethodView):
    MSG_DUPLICATES_RECORD = 'Duplicates another record.'
    MSG_CNAME_CONFLICT = 'CNAME record conflicts with another ones having the same name.'

    @supports_pagination
    @render_page(RecordSchema)
    @punycode_domain_name
    def get(self, domain_name, pageable):
        self._require_permission(domain_name)

        return self.record_repository.find_paged_by_domain_name(
            pageable=pageable,
            domain_name=domain_name,
        )

    @master_session
    @parse_body(
        RecordSchema,
        partial=['id'],
        provider=record_provider,
    )
    @render(RecordSchema, status_code=201)
    @punycode_domain_name
    def post(self, domain_name, body):
        self._require_permission(domain_name)

        is_new = body not in self.record_repository.session
        try:
            domain = body.domain or self.domain_repository.find_by_name(domain_name)
        except sqlalchemy.orm.exc.NoResultFound:
            abort(404, message='Domain not found', code='domain_not_found')
        else:
            body.domain = domain

        try:
            self._check_conflicts(body)
            self._disable_domain_pdd_sync(body.domain)
            record = self.record_repository.save(body, flush=True)
        except sqlalchemy.exc.IntegrityError as e:
            self._handle_integrity_error(e)
            raise
        else:
            if is_new:
                return record
            else:
                return record, 200

    def _check_conflicts(self, record):
        if record.type == RecordType.CNAME:
            filter = (Record.name == record.name)
            filter = filter & (Record.domain_id == record.domain.id)

            if record.id is not None:
                filter = filter & (Record.id != record.id)

            if self.record_repository.find_first(filter) is not None:
                abort(
                    422,
                    message='Invalid request body',
                    errors=dict(_schema=[self.MSG_CNAME_CONFLICT]),
                    code='cname_record_conflict',
                )
        else:
            filter = (Record.name == record.name)
            filter = filter & (Record.domain_id == record.domain.id)
            filter = filter & (Record.type == RecordType.CNAME)

            if record.id is not None:
                filter = filter & (Record.id != record.id)

            if self.record_repository.find_first(filter) is not None:
                abort(
                    422,
                    message='Invalid request body',
                    errors=dict(_schema=[self.MSG_CNAME_CONFLICT]),
                    code='cname_record_conflict',
                )

    def _handle_integrity_error(self, e):
        duplication = (
                e.orig.pgcode == '23505' and
                e.orig.diag.constraint_name == 'idx_uniq_record'
        )
        if duplication:
            abort(
                422,
                message='Invalid request body',
                errors=dict(_schema=[self.MSG_DUPLICATES_RECORD]),
                code='invalid_request_body',
            )


class RecordDetailsView(DomainMethodView):
    @render(RecordSchema)
    @punycode_domain_name
    def get(self, domain_name, record_id):
        self._require_permission(domain_name)

        try:
            record = self.record_repository.find_by_domain_name_and_id(
                domain_name,
                record_id,
            )
        except sqlalchemy.orm.exc.NoResultFound:
            abort(404, message='Record not found', code='record_not_found')
        else:
            return record

    @master_session
    @render(RecordSchema)
    @punycode_domain_name
    def delete(self, domain_name, record_id):
        self._require_permission(domain_name)

        try:
            record = self.record_repository.find_by_domain_name_and_id(
                domain_name,
                record_id,
            )
        except sqlalchemy.orm.exc.NoResultFound:
            abort(404, message='Record not found', code='record_not_found')
        else:
            self._disable_domain_pdd_sync(record.domain)
            self.record_repository.delete(record)
            return record


domains_bp = flask.Blueprint('domains', __name__)

domains_bp.add_url_rule(
    '/',
    view_func=DomainCollectionView.as_view('domain_collection')
)

domains_bp.add_url_rule(
    '/<domain_name>/',
    view_func=DomainDetailsView.as_view('domain_details')
)

domains_bp.add_url_rule(
    '/<domain_name>/records/',
    view_func=RecordCollectionView.as_view('record_collection')
)

domains_bp.add_url_rule(
    '/<domain_name>/records/<int:record_id>/',
    view_func=RecordDetailsView.as_view('record_details')
)
