# encoding: UTF-8

import flask.views
import ylog.context
from ws_properties.utils.logs import get_logger_for_instance

from appcore.data.session import master_session
from appcore.injection import Injected
from appcore.security.decorators import authenticated_client
from dns_hosting.dao.domains import DomainRepository
from dns_hosting.models.domains import Domain
from dns_hosting.services.auth.model import Scope


class EventListenerView(flask.views.MethodView):
    decorators = (
        authenticated_client(scopes=Scope.fire_events, fail_with_404=True),
    )

    domain_repository = Injected('domain_repository')  # type: DomainRepository

    def __init__(self, *args, **kwargs):
        super(EventListenerView, self).__init__(*args, **kwargs)
        self._logger = get_logger_for_instance(self)

    @master_session
    def post(self):
        event = flask.request.json

        with ylog.context.log_context(event=event):
            self._logger.info('Got event sent by directory.')

        event_name = event['event']
        event_data = event['object']
        event_revision = event['revision']

        if event_name in {'domain_deleted', 'domain_alienated'}:
            self.on_domain_removed(event_data, event_revision)

        elif event_name == 'organization_deleted':
            self.on_organization_removed(event_data, event_revision)

        else:
            self._logger.warning('Unsupported event. Skip it.')

        return flask.jsonify()

    def on_domain_removed(self, data, revision):
        if not data['owned']:
            self._logger.info('Domain not owned. Skip it.')
            return

        domain = self.domain_repository.find_by_name(data['name'])
        if domain:
            if domain.org_id is None or domain.org_id == data['org_id']:
                self.domain_repository.delete(domain)
                self._logger.info('Domain \'%s\' removed.', domain.name)
            else:
                self._logger.warning('Domain organizations don\'t match.')
                return
        else:
            self._logger.warning('Domain doesn\'t exist.')

    def on_organization_removed(self, organization, revision):
        domains = self.domain_repository.find_by_org_id(organization['id'])
        for domain in domains:
            self.domain_repository.delete(domain)


events_bp = flask.Blueprint('events', __name__)

events_bp.add_url_rule(
    '/',
    view_func=EventListenerView.as_view('event_listener'),
)
