#!/usr/bin/python
# -*- coding: UTF-8 -*-

"""
    Необходимые переменные окружения:
        PGHOST - для прода специальный RO FQDN, для теста единственный инстанс БД
        PGDATABASE
        SLEEP_SECONDS = перерыв между запусками
"""


import os, logging, time

import psycopg2

from dns_hosting.services.dnsserver.nsd.ctl import NSDControl

logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)

ch = logging.StreamHandler()
ch.setLevel(logging.INFO)

formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
ch.setFormatter(formatter)

logger.addHandler(ch)

logger.info("Zonetracker setting up")


DB_CONFIG = {
    "host":         os.environ['PGHOST'],
    "port":         os.environ.get('PGPORT', '6432'),
    "dbname":       os.environ['PGDATABASE'],
    "user":         os.environ.get('PGUSER', 'internal_user'),
    "sslmode":      "verify-full",
}

NSDCONTROL_CONFIG = {
    "address":  ("::1", 8952),
    "keyfile":  "/etc/nsd/nsd_control.key",
    "certfile": "/etc/nsd/nsd_control.pem",
}

MASTER_ZONES_STMT = "SELECT name FROM domains"

SLEEP_SECONDS = int(os.environ.get('SLEEP_SECONDS', '60'))

def config_not_empty(config):
    for key, value in config.items():
        if not value:
            raise ValueError("Missing config value for key: " + key)

def setup_db_conn(config):

    config_not_empty(config)
    logger.info("Setting up connection to %s", config['dbname'])
    conn = psycopg2.connect(**config)

    return conn

def setup_NSDControl(config):

    config_not_empty(config)
    logger.info("Setting up connection to nsd")
    nsd_control = NSDControl(**config)

    return nsd_control

def main():

    logger.info("Zonetracker starting iteration")

    starttime = time.time()

    db_connection = setup_db_conn(DB_CONFIG)
    nsd_control = setup_NSDControl(NSDCONTROL_CONFIG)

    with db_connection.cursor(name='zonetracker_fetch') as cursor:
        logger.info("Fetching zones from master")
        cursor.execute(MASTER_ZONES_STMT)
        master_zones = { item[0] for item in cursor };

    db_connection.close()

    logger.info("Got zones from master: %s", len(master_zones))

    if len(master_zones)==0:
        raise ValueError("Illegal value, got no zones from master")

    logger.info("Fetching zones from nsd")
    nsd_zones = { item['zone'] for item in nsd_control.zonestatus() }

    logger.info("Got zones from nsd: %s", len(nsd_zones))

    # TODO: не держать в памяти два сета, а вычислять разницу потоково
    zones_to_add = master_zones.difference(nsd_zones)
    zones_to_delete = nsd_zones.difference(master_zones)

    logger.info("Found new zones: %s", len(zones_to_add))
    logger.info("Found old zones: %s", len(zones_to_delete))

    # action
    if zones_to_add:
        logger.info("Adding new zones to nsd")
        nsd_control.addzones(zones_to_add)

    if zones_to_delete:
        logger.info("Deleting old zones from nsd")
        nsd_control.delzones(zones_to_delete)

    finishtime = time.time()
    elapsed_seconds = finishtime - starttime
    logger.info("Iteration took seconds - %s", elapsed_seconds)
    logger.info("Zonetracker finished iteration")

if __name__ == '__main__':
    while True:

        try:
            main()
        except Exception:
            logger.exception("Exception, exiting")
            exit(1)

        time.sleep(SLEEP_SECONDS)
