import asyncio
from typing import Optional

from loguru import logger
from sqlalchemy import text
from sqlalchemy.ext.asyncio import AsyncEngine, AsyncSession, create_async_engine, AsyncConnection
from sqlalchemy.orm import sessionmaker

from exceptions import DBConnectionError
from settings import config, DBSettings
from utils.common import get_engine_url, HostManager


class WorkspaceDnsDB:
    def __init__(self, settings: DBSettings):
        self._settings = settings
        logger.info('Initiating database')
        # мастер пока задаем в параметрах, а реплики выбираем ближайшие по ДЦ
        self.host_manager = HostManager(self._settings)
        self.create_session_maker()

    def create_session_maker(self):
        self._engine_master_url = get_engine_url(self._settings, host=self.host_manager.get_master_host())
        self._engine_replica_url = get_engine_url(self._settings, host=self.host_manager.get_replica_host())

        self._engine_master = self._create_engine(
            master=True,
            connect_args={
                'statement_cache_size': 200,
            },
        )
        self._engine_replica = self._create_engine(
            master=False,
            connect_args={
                'statement_cache_size': 200,
            },
        )
        self._session_maker_master = self._create_session_maker(master=True)
        self._session_maker_replica = self._create_session_maker(master=False)

    @property
    def settings(self) -> DBSettings:
        return self._settings

    @property
    def engine_master_url(self) -> str:
        return self._engine_master_url

    @property
    def engine_replica_url(self) -> str:
        return self._engine_replica_url

    @property
    def engine_master(self):
        return self._engine_master

    @property
    def engine_replica(self):
        return self._engine_replica

    @property
    def session_master(self):
        return self._session_maker_master

    @property
    def session_replica(self):
        return self._session_maker_replica

    async def _dispose(self) -> None:
        await self.engine_master.dispose()
        await self.engine_replica.dispose()

    async def close_db(self) -> None:
        self.session_master.close_all()
        self.session_replica.close_all()

    def _create_engine(
        self,
        master: bool,
        connect_args: Optional[dict] = None,
    ) -> AsyncEngine:
        return create_async_engine(
            self.engine_master_url if master else self.engine_replica_url,
            pool_size=self._settings.pool_size,
            max_overflow=self._settings.pool_max_overflow,
            echo=self._settings.echo,
            echo_pool=self._settings.pool_echo,
            connect_args=connect_args,
        )

    def _create_session_maker(self, master: bool) -> sessionmaker:
        return sessionmaker(
            self.engine_master if master else self.engine_replica,
            expire_on_commit=False,
            autocommit=False,
            class_=AsyncSession,
        )

    def recreate_session_maker(self):
        # можно потом заиспользовать pg_pinger https://a.yandex-team.ru/arc/trunk/arcadia/toolbox/pg-pinger
        logger.info('recreate session to DB')
        self.create_session_maker()


db = WorkspaceDnsDB(config.db)


async def get_session_maker_master() -> AsyncConnection:
    session_maker = db.session_master
    retry_attempt = 0
    while retry_attempt != db.settings.connect_retry:
        try:
            async with session_maker.begin() as session:
                # await session.execute(text('select 1;'))
                return session
        except Exception as exc:
            logger.debug(
                f"DB connection lost due to {exc} - sleeping for 0.001 sec and will retry (attempt #{retry_attempt})"
            )
            await asyncio.sleep(0.001)
        retry_attempt += 1

    raise DBConnectionError


async def get_session_maker_replica() -> AsyncConnection:
    session_maker = db.session_replica
    retry_attempt = 0
    while retry_attempt != db.settings.connect_retry:
        try:
            async with session_maker.begin() as session:
                await session.execute(text('select 1;'))
                return session
        except Exception as exc:
            logger.debug(
                f"DB connection lost due to {exc} - sleeping for 0.001 sec and will retry (attempt #{retry_attempt})"
            )
            await asyncio.sleep(0.001)
        retry_attempt += 1
    db.recreate_session_maker()
    raise DBConnectionError
