import enum
import uuid

import sqlalchemy as sa
import sqlalchemy.orm as orm
import sqlalchemy.types as types
from dns import rdata
from dns import rdataclass
from dns import rdatatype
from sqlalchemy import MetaData
from sqlalchemy.ext.declarative import as_declarative

metadata = MetaData()


@as_declarative(metadata=metadata)
class Base:
    pass


class UUID(types.TypeDecorator):
    impl = types.CHAR

    def process_bind_param(self, value, dialect):
        if value is None:
            return None
        else:
            return value.hex

    def process_result_value(self, value, dialect):
        if value is None:
            return None
        else:
            return uuid.UUID(hex=value)


def get_values(e):
    return e.__members__.values()


class Domain(Base):
    __tablename__ = 'domains'

    id = sa.Column(sa.Integer, primary_key=True)
    name = sa.Column(sa.String)
    serial = sa.Column(sa.Integer)
    org_id = sa.Column(sa.Integer)
    revision = sa.Column(sa.BigInteger)
    is_technical = sa.Column(sa.Boolean)
    pdd_sync_enabled = sa.Column(sa.Boolean)
    pdd_domain_id = sa.Column(sa.Integer)

    records = orm.relation(
        lambda: Record,
        uselist=True,
        cascade='merge, expunge, delete, delete-orphan',
        back_populates='domain',
    )

    def __init__(
        self,
        id=None,
        serial=None,
        name=None,
        org_id=None,
        revision=None,
        is_technical=False,
        pdd_sync_enabled=True,
        pdd_domain_id=None,
    ):
        self.id = id
        self.serial = serial
        self.name = name
        self.org_id = org_id
        self.revision = revision
        self.is_technical = is_technical
        self.pdd_sync_enabled = pdd_sync_enabled
        self.pdd_domain_id = pdd_domain_id


class RecordType(str, enum.Enum):
    A = 'A'
    AAAA = 'AAAA'
    CNAME = 'CNAME'
    MX = 'MX'
    TXT = 'TXT'
    SRV = 'SRV'
    NS = 'NS'
    CAA = 'CAA'


class Record(Base):
    __tablename__ = 'records'

    id = sa.Column(sa.Integer, primary_key=True)
    domain_id = sa.Column(sa.ForeignKey(Domain.id))
    name = sa.Column(sa.String)
    type = sa.Column(sa.Enum(RecordType))
    content = sa.Column(sa.String)
    ttl = sa.Column(sa.Integer)

    @property
    def rdata(self):
        return rdata.from_text(
            rdataclass.IN,
            rdatatype.from_text(self.type),
            self.content,
        )

    @rdata.setter
    def rdata(self, rdata):
        self.content = rdata.to_text()

    domain = orm.relation(
        lambda: Domain,
        cascade='save-update, merge',
        back_populates='records',
    )

    def __init__(
        self,
        id=None,
        domain_id=None,
        name=None,
        type=None,
        content=None,
        ttl=None,
    ):
        self.id = id
        self.domain_id = domain_id
        self.name = name
        self.type = type
        self.content = content
        self.ttl = ttl


class Operation(str, enum.Enum):
    DOMAIN_ADD = 'domain-add'
    DOMAIN_DELETE = 'domain-delete'
    DOMAIN_CLEAR = 'domain-clear'
    RECORD_ADD = 'record-add'
    RECORD_DELETE = 'record-delete'
    RECORD_UPDATE_TTL = 'record-update-ttl'


class ChangeLog(Base):
    __tablename__ = 'change_log'

    version = sa.Column(UUID, primary_key=True)
    origin = sa.Column(sa.String)
    serial = sa.Column(sa.Integer)
    operation = sa.Column(sa.Enum(Operation, values_callable=get_values))
    name = sa.Column(sa.String)
    type = sa.Column(sa.Enum(RecordType))
    content = sa.Column(sa.String)
    ttl = sa.Column(sa.Integer)

    def __init__(
        self,
        version=None,
        origin=None,
        serial=None,
        operation=None,
        name=None,
        type=None,
        content=None,
        ttl=None,
    ):
        self.version = version
        self.origin = origin
        self.serial = serial
        self.operation = operation
        self.name = name
        self.type = type
        self.content = content
        self.ttl = ttl
