import json
from typing import Generator

from aiohttp import web
from lib.sendr_qstats.http.aiohttp import get_registry_handler
from setproctitle import setproctitle
from sqlalchemy import select
from sqlalchemy.ext.asyncio import AsyncConnection

from database.db import get_session_maker_replica
from models.domain import Domain
from settings import config
from utils.common import IteratorJSONAdapter
from utils.stat import REGISTRY, get_stats_middleware

json_encoder = json.JSONEncoder()


async def get_domain_names(session: AsyncConnection, yield_per: int = 10000) -> Generator:
    async with session.begin():
        query = select(Domain.name).order_by(Domain.name.asc())
        result = (await session.execute(query, execution_options={"server_side_cursor": True})).yield_per(yield_per)
    return (row[0] for row in result)


async def http_master(request):
    session = await get_session_maker_replica()
    response = web.StreamResponse(
        status=200,
        reason='OK',
    )
    response.headers["Content-Type"] = "application/json"
    await response.prepare(request)

    names_it = await get_domain_names(session)
    adapted_names_it = IteratorJSONAdapter(names_it)
    for chunk in json_encoder.iterencode(adapted_names_it):
        await response.write(chunk.encode('utf-8'))

    await response.write_eof()
    return response


async def ping(request):
    return web.json_response(text="pong")



def init_app():
    routes = [
        web.get('/ping', ping),
        web.get('/unistat', get_registry_handler(REGISTRY))
    ]
    middlewares = []

    if config.http_master:
        setproctitle("dns_master_http")
        routes.append(web.get('/', http_master))
        middlewares.append(get_stats_middleware(REGISTRY))

    app = web.Application(
        middlewares=middlewares,
    )

    app.add_routes(routes)
    return app
