import logging
from sys import stderr

from loguru import logger, _better_exceptions
from ujson import dumps

from settings import config


class InterceptHandler(logging.Handler):
    # https://loguru.readthedocs.io/en/stable/overview.html?highlight=InterceptHandler#entirely-compatible-with-standard-logging
    def emit(self, record):
        try:
            level = logger.level(record.levelname).name
        except ValueError:
            level = record.levelno

        frame, depth = logging.currentframe(), 2
        while frame.f_code.co_filename == logging.__file__:
            frame = frame.f_back
            depth += 1

        logger.opt(depth=depth, exception=record.exc_info).log(level, record.getMessage())


def deploy_formatter(record):
    record["extra"]["serialized"] = dumps(
        {
            "@timestamp": record["time"].isoformat(),
            "message": record["message"],
            "stackTrace": "".join(
                _better_exceptions.ExceptionFormatter().format_exception(
                    record["exception"].type, record["exception"].value, record["exception"].traceback
                )
            )
            if record["exception"]
            else None,
            "levelStr": record["level"].name,
            "level": record["level"].no,
            "loggerName": f"{record['name']}:{record['function']}",
            # extra
            "line": record["line"],
            "threadName": record["file"].path,
        }
    )
    return "{extra[serialized]}\n"


def initialize_logging():
    logger.remove()
    if config.logging.stdout_enabled:
        format = {"format": deploy_formatter} if config.logging.stdout_json_formatted else {}
        logger.add(stderr, enqueue=True, **format)
    if config.logging.file_enabled:
        logger.add(
            config.logging.file_path,
            enqueue=True,
            rotation=config.logging.file_rotation_rule,
            retention=config.logging.file_retention_count,
            compression="gz",
        )

    logging.basicConfig(handlers=[InterceptHandler()], level=0)
