import contextlib
import time
from typing import Callable

from lib.sendr_qstats import MetricsRegistry, Histogram, Counter, Gauge, registry

from settings import config

REGISTRY = MetricsRegistry(ctype=config.env)

STATS_RESPONSE_TIME_BUCKETS = (
    0.0, .005, .01,
    .025, .05, .075, .1,
    .25, .5, .75, 1.0,
    2.5, 5.0, 7.5, 10.0, 12.5, 15, 17.5, 20,
    25, 30, 60
)


dns_rcode = Counter(
    'rcode',
    labelnames=('rcode', ),
    registry=REGISTRY,
)

dns_opcode = Counter(
    'opcode',
    labelnames=('query', ),
    registry=REGISTRY,
)

dns_request_type = Counter(
    'rdtype',
    labelnames=('rdtype', ),
    registry=REGISTRY,
)

tcp_connect_count_gauge = Gauge(
    'tcp_connect',
    labelnames=('count', ),
    registry=REGISTRY,
)

tcp_connect_count = Counter(
    'tcp_connect',
    labelnames=('reason', ),
    registry=REGISTRY,
)

# TODO: повесть на handle_dns декоратор
# dns_request_time = Histogram(
#     'query_time',
#     labelnames=('query_time', ),
#     registry=REGISTRY,
# )

tcp_connect_time = Histogram(
    'tcp_connect_time',
    labelnames=('tcp_connect_time', ),
    registry=REGISTRY,
)


def metrics(func):
    async def wrap_func(*args, **kwargs):
        start = time.perf_counter()
        tcp_connect_count_gauge.labels('count').inc()
        try:
            await func(*args, **kwargs)
        except Exception as e:
            tcp_connect_count.labels('error').inc()
            raise e
        tcp_connect_time.labels('tcp_connect_time').observe(time.perf_counter() - start)
        tcp_connect_count.labels('success').inc()
        tcp_connect_count_gauge.labels('count').dec()
    return wrap_func

# Так как роуты не используем, то пришлось чуть модифицировать функцию , а не брать из либы qstats
def get_stats_middleware(handle_registry: MetricsRegistry = registry.REGISTRY) -> Callable:
    from aiohttp import web

    REQUEST_TIME = Histogram(
        'request_time',
        labelnames=('route_name', 'method'),
        registry=handle_registry,
    )

    # timings of handled requests - handler returned response object
    REQUEST_HANDLE_TIME = Histogram(
        'request_handle_time',
        labelnames=('route_name', 'method'),
        registry=handle_registry,
    )

    # counter of handled requests - handler returned response
    REQUEST_STATUS_COUNT = Counter(
        'request_status',
        labelnames=('route_name', 'method', 'status'),
        registry=handle_registry,
    )

    # number of requests being handled by application at the moment
    REQUEST_COUNT_GAUGE = Gauge(
        name='request_count',
        labelnames=('route_name', 'method'),
        registry=handle_registry,
    )

    @web.middleware
    async def middleware_stats(request, handler):
        """
        Middleware для автоматической инструментации обработки запросов
        """

        route = request.match_info.route
        route_name = route._resource._path or 'unknown'
        method = request.method

        if route_name in ['unistat']:
            return await handler(request)

        with contextlib.ExitStack() as ctx:
            ctx.enter_context(REQUEST_TIME.labels(route_name, method).time)
            ctx.enter_context(REQUEST_COUNT_GAUGE.labels(route_name, method).track)

            start = time.perf_counter()

            response = await handler(request)

            REQUEST_HANDLE_TIME.labels(route_name, method).observe(time.perf_counter() - start)

            status = str(response.status)
            generic_status = f'{status[0]}xx'
            REQUEST_STATUS_COUNT.labels(route_name, method, status).inc()
            REQUEST_STATUS_COUNT.labels(route_name, method, generic_status).inc()

        return response

    return middleware_stats
