import asyncio
import os
from contextlib import asynccontextmanager
from typing import Set

from loguru import logger

from nsd.nsd_control import NSDControl
from nsd.zone_tracker import ZoneTracker
from settings import config
from utils.common import pidfile

nsdctrl = NSDControl()
zonetrckr = ZoneTracker()

async def get_stuck_zones() -> Set:
    stucked_zones = set()
    async for zone in nsdctrl.zonestatus():
        # тут считаем только те зоны которые не записали в БД
        if zone.commit_serial is None and zone.served_serial is None:
            stucked_zones.add(zone.name)

    return stucked_zones


@asynccontextmanager
async def running_nsd():
    init_conf_file = f"/app/config/nsd-init-{config.env}.conf"
    if not os.path.exists(init_conf_file):
        init_conf_file = "/etc/nsd/nsd.conf"

    proc = await asyncio.create_subprocess_exec(
        "nsd", "-d", "-c", init_conf_file,
        stdout=asyncio.subprocess.PIPE
    )
    try:
        yield proc
    finally:
        logger.info("shutting down NSD")
        proc.terminate()


@pidfile(path=config.init_pid_path)
async def init_nsd_db():
    logger.info("start NSD")
    async with running_nsd():
        logger.info("start getting local zones")
        await zonetrckr.update_local_zones()
        logger.info(f"local zones count {len(zonetrckr.local_names_it)}")

        # зон в http master не может быть 0, поэтому если мастер выкинет ошибку - перезапрашиваем снова
        while not zonetrckr.remote_names_it:
            logger.info("start getting zones from http master")
            await zonetrckr.update_remote_zones()
            logger.info(f"remote zones count {len(zonetrckr.remote_names_it)}")

        logger.info("start updating the zones")
        await zonetrckr.actualize_zones(nsdctrl=nsdctrl)
        logger.info("finish zones updating")

        stuck_zones = await get_stuck_zones()
        logger.info(f"stuck zones count {len(stuck_zones)}")

        while len(stuck_zones) > config.init_stuck_zones_count:
            logger.info(f"waiting {config.init_sleep_time} sec. for downloading zones...")
            await asyncio.sleep(config.init_sleep_time)
            # приходится хранить два сета в памяти
            new_stuck_zones = await get_stuck_zones()
            logger.info(f"stuck zones count {len(new_stuck_zones)}")
            if len(new_stuck_zones) < len(stuck_zones):
                stuck_zones = new_stuck_zones
                continue
            # если зоны залипли и не насасываются по какой-то причине, то передобавляем их
            logger.info('fixing stuck zones')
            await nsdctrl.delzones(list(new_stuck_zones))
            await nsdctrl.addzones(list(new_stuck_zones))
            stuck_zones = new_stuck_zones


        logger.info("init finished")
