from typing import List

from aiohttp import web
from loguru import logger

from settings import Settings
from utils.common import _Handlers


class Application:
    def __init__(self, config: Settings, handlers: _Handlers, tasks: List):
        self._handlers = handlers
        self._host = config.web.host
        self._port = config.web.port

        self._app = web.Application(
            middlewares=[web.normalize_path_middleware()]
        )

        self._tasks = tasks

        async def on_shutdown(_):
            for task in self._tasks:
                task.cancel()

        self._app.on_shutdown.append(on_shutdown)
        self._init_routes()

    def _init_routes(self):
        app = self._app

        for handler in self._handlers:
            method = handler[0]
            path = handler[1]
            coro = handler[2]

            app.router.add_route(method, path, coro)
            logger.debug(f'initialized handler; method={method}, path={path}')

        app.router.add_route('GET', '/ping', self.ping)

    async def ping(self, request):
        return web.Response(text='pong', status=200)

    @property
    def port(self):
        return self._port

    @property
    def host(self):
        return self._host

    def run(self):
        return web.AppRunner(self._app)
