import argparse
import asyncio
import sys

from aiohttp import web
from loguru import logger

from actualize_zones import init_nsd_db
from app import Application
from nsd.nsd_manager import NSDManager
from settings import config
from utils.common import make_periodic
from utils.logconf import initialize_logging


def start_tasks(tasks, loop) -> list:
    _tasks = []

    for delay, coro in tasks:
        coro = make_periodic(delay, coro)
        task = loop.create_task(coro())
        _tasks.append(task)
    return _tasks

def build_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--init", action="store_true", help="run initialize NSD db")
    return parser.parse_args()


def main():
    initialize_logging()
    args = build_args()
    loop = asyncio.get_event_loop()

    if args.init and not config.skip_update_db:
        logger.info("start init NSD db")
        loop.run_until_complete(init_nsd_db())
        sys.exit(0)

    nsd_manager = NSDManager(config)
    tasks = start_tasks(nsd_manager.tasks, loop)

    app = Application(config, nsd_manager.handlers, tasks)
    runner =  app.run()
    loop.run_until_complete(runner.setup())
    site = web.TCPSite(runner=runner, host=app.host, port=app.port)
    loop.run_until_complete(site.start())

    try:
        loop.run_forever()
    except (KeyboardInterrupt, SystemExit):
        # TODO: отлавливать сигнал sigterm
        logger.info("stop nsd manager")
    finally:
        loop.stop()


if __name__ == "__main__":
    main()
