import asyncio
import os.path
from datetime import timedelta

import aiofiles
from aiohttp import web
from loguru import logger
from requests import Response

from nsd.nsd_control import NSDControl
from nsd.zone_tracker import ZoneTracker
from settings.base import Settings
from utils.common import BaseModule


class NSDManager(BaseModule):
    def __init__(self, config: Settings):
        super().__init__(config)
        self.nsd_control = NSDControl()
        self.zonetrckr = ZoneTracker()
        self.config = config

        self._stucked_zones = []
        self._current_stats = []
        self._current_zonestatus = []
        self._current_tcp6_fd_stats = []

    @property
    def handlers(self):
        return (
            ("GET", "/unistat", self.unistat),
        )

    @property
    def tasks(self):
        return (
            (timedelta(seconds=5), self.stat_server),
            (timedelta(minutes=15), self.zonetracker),
            (timedelta(minutes=15), self.zonefixer),
            (timedelta(minutes=40), self.update_zonestatus),
        )

    async def unistat(self, request) -> Response:
        return web.json_response(
            self._current_tcp6_fd_stats +
            self._current_stats +
            self._current_zonestatus
        )

    async def zonetracker(self):
        await self.zonetrckr.update_local_zones()
        logger.info(f"local zones count {len(self.zonetrckr.local_names_it)}")
        logger.info("start getting zones from http master")
        await self.zonetrckr.update_remote_zones()
        logger.info(f"remote zones count {len(self.zonetrckr.remote_names_it)}")

    async def zonefixer(self):
        if os.path.exists(self.config.init_pid_path):
            logger.info("zonefixer skipped - init nsd in progress")
            return

        if self.zonetrckr.remote_names_it:
            logger.info("start updating the zones")
            await self.zonetrckr.actualize_zones(nsdctrl=self.nsd_control)
            logger.info("zones is updating")
        if self._stucked_zones:
            chunk_size = 100000
            try:
                logger.info(f"fixing {len(self._stucked_zones)} stucked zones...",)
                for offset in range(0, len(self._stucked_zones), chunk_size):
                    chunk = self._stucked_zones[offset:offset + chunk_size]
                    await self.nsd_control.delzones(chunk)
                    await self.nsd_control.addzones(chunk, "default")
                    await asyncio.sleep(0.5)

            except Exception as e:
                logger.error(f"failed to fix stucked zones: {e}")
            else:
                logger.info("stucked zones fixed.")
                self._stucked_zones = []


    async def stat_server(self):
        await self._set_current_stats()
        await self._set_current_tcp6_fd_stats()

    async def _set_current_tcp6_fd_stats(self):
        try:
            async with aiofiles.open("/proc/net/tcp6", mode="r") as f:
                tcp6_fd_count = len(await f.readlines()) - 1
        except Exception as e:
            logger.error(f"TCP6 FD stat aggregation failed: {e}")
            self._current_tcp6_fd_stats = []
        else:
            self._current_tcp6_fd_stats = [
                ["tcp6_fd_count_ammx", tcp6_fd_count],
            ]
            logger.info("TCP6 FD stat loaded.")

    async def _set_current_stats(self):
        try:
            data = await self.nsd_control.stats()
        except Exception as e:
            logger.error(f"stat aggregation failed: {e}")
            self._current_stats = []
            return

        stats = []
        for line in data.decode().splitlines():
            line = line.strip()
            if not line:
                continue

            key, value = line.split("=")
            key = key.replace(".", "_")
            if "time_" in key:
                key = key + "_avvv"
                value = float(value)
            elif "size_" in key:
                key = key + "_ammv"
                value = int(value)
            elif "zone_" in key:
                key = key + "_annx"
                value = int(value)
            else:
                key = key + "_ammm"
                value = int(value)
            stats.append([key, value])

        self._current_stats = stats
        logger.info("NSD stat loaded.")

    async def update_zonestatus(self):
        await self._set_current_zonestatus()

    async def _set_current_zonestatus(self):
        logger.info("Zonestatus loading...")
        stucked_zones = set()
        stucked_zones_loaded = False
        while not stucked_zones_loaded:
            try:
                async for zone in self.nsd_control.zonestatus():
                    if self.config.tcp_fd_stuck and zone.transfer == "waiting-for-TCP-fd":
                            stucked_zones.add(zone.name)
                    if zone.commit_serial is None and zone.served_serial is None:
                        stucked_zones.add(zone.name)
            except ConnectionRefusedError:
                # если NSD не запущен, пробуем снова
                logger.error(f"failed connect to NSD, retry for 5 sec.")
                await asyncio.sleep(5)
                continue
            except Exception as e:
                # в остальных случаях кидаем считаем за ошибку
                logger.error(f"zonestatus aggregation failed: {e}")
                self._current_zonestatus = []
                stucked_zones_loaded = True
            else:
                self._current_zonestatus = [["stucked_zones_ammx", len(stucked_zones)]]
                logger.info("zonestatus loaded.")
                stucked_zones_loaded = True
                if stucked_zones:
                    self._stucked_zones = list(stucked_zones)

