import os
from typing import Iterable, List

import aiofiles
import aiohttp
import ijson
from loguru import logger

from nsd.nsd_control import NSDControl
from settings import config


def safe_next(it):
    try:
        return next(it)
    except StopIteration:
        return None

# пережиток python2 - чтобы не менять логику ijoin
def cmp(a, b):
    return (a > b) - (a < b)

def ijoin(it1: Iterable, it2: Iterable, cmp=cmp, default=None):
    it1, it2 = iter(it1), iter(it2)
    v1, v2 = safe_next(it1), safe_next(it2)

    while v1 is not None and v2 is not None:
        value = cmp(v1, v2)
        if value < 0:
            yield v1, default
            v1 = safe_next(it1)
        elif value > 0:
            yield default, v2
            v2 = safe_next(it2)
        else:
            yield v1, v2
            v1, v2 = safe_next(it1), safe_next(it2)

    while v1 is not None:
        yield v1, default
        v1 = safe_next(it1)

    while v2 is not None:
        yield default, v2
        v2 = safe_next(it2)


class ZoneTracker:
    def __init__(self):
        self.dns_master_url = config.dns_master_url
        self.zone_list = config.zone_list_path
        self.local_names_it = []
        self.remote_names_it = []

    async def get_remote_zones(self) -> Iterable[str]:
        async with aiohttp.request("get", url=self.dns_master_url) as r:
            content = await r.content.read()
            return ijson.parse(content)

    async def update_remote_zones(self):
        self.remote_names_it = [value
                            for prefix, event, value in await self.get_remote_zones()
                            if event == "string"
                        ]

    async def update_local_zones(self):
        if not os.path.exists(self.zone_list):
            return

        zone_names = []
        async with aiofiles.open(self.zone_list, mode="r") as f:
            async for line in f:
                if line.startswith("#"):
                    continue
                action, zone_name, pattern = line.split(" ", 3)
                if action == "add":
                    zone_names.append(zone_name)
        zone_names.sort()
        self.local_names_it = zone_names

    async def apply_changes(self, added: List, removed: List, nsdctrl: NSDControl):
        if added:
            await nsdctrl.addzones(added)
            logger.info(f"added zones count: {len(added)}")
        if removed:
            await nsdctrl.delzones(removed)
            logger.info(f"removed zones count: {len(removed)}")

    async def actualize_zones(self, nsdctrl: NSDControl):
        added = []
        removed = []
        for local_name, remote_name in ijoin(self.local_names_it, self.remote_names_it):
            if local_name is None and remote_name is None:
                raise AssertionError("Illegal state error.")
            elif remote_name is None:
                removed.append(local_name)
            elif local_name is None:
                added.append(remote_name)
            else:
                pass  # nothing to do

        if added or removed:
            await self.apply_changes(added, removed, nsdctrl)

