import asyncio
import functools
import os
import ssl
from datetime import timedelta
from typing import Awaitable, Coroutine, Tuple, Union

import aiofiles
from loguru import logger

from settings import config, Settings

_Method = str
_Path = str
_Handlers = Tuple[Tuple[_Method, _Path, Awaitable]]

_Tasks = Tuple[Tuple[timedelta, Coroutine]]


class BaseModule:
    def __init__(self, config: Settings):
        pass

    @property
    def handlers(self) -> _Handlers:
        return tuple()

    @property
    def tasks(self) -> _Tasks:
        return tuple()


def make_periodic(delay: Union[int, timedelta], coro):
    """Возвращает новую корутину, которая будет запускать себя снова и снова.
    """
    if isinstance(delay, timedelta):
        delay = delay.total_seconds()

    @functools.wraps(coro)
    async def periodic():
        nonlocal delay

        while True:
            try:
                logger.debug(f'started cycle, coro: {coro.__name__}')
                await coro()
            except asyncio.CancelledError:
                delay = 0  # stop
            except Exception as e:
                logger.exception(f'failed, coro: {coro.__name__}; error: {e}')

            if not delay:
                logger.debug(f'stopped, coro: {coro.__name__}')
                return

            try:
                logger.debug(f'sleep coro: {coro.__name__}, delay={delay}')
                await asyncio.sleep(delay)
            except asyncio.CancelledError:
                logger.debug('stopped')
                return

    return periodic



def ssl_context():
    ssl_ctx = ssl.SSLContext(ssl.PROTOCOL_TLS_CLIENT)
    ssl_ctx.load_cert_chain(config.nsd.crt, keyfile=config.nsd.key)
    ssl_ctx.check_hostname = False
    ssl_ctx.verify_mode = ssl.VerifyMode.CERT_NONE
    return ssl_ctx

ssl_ctx = ssl_context()


class PidFile():
    def __init__(self, path: str):
        self.filename = path
        self.pid = os.getpid()

    async def create(self):
        logger.debug(f"create pidfile: {self.filename}")
        async with aiofiles.open(self.filename, "a+") as f:
            await f.write(f"{self.pid}\n")

    def close(self):
        if self.filename and os.path.isfile(self.filename):
            logger.debug(f"remove pidfile: {self.filename}")
            os.remove(self.filename)

    async def __aenter__(self):
        await self.create()
        return self

    async def __aexit__(self, exc_type=None, exc_value=None, exc_tb=None):
        self.close()

def pidfile(path):
    def wrapper(func):
        @functools.wraps(func)
        async def decorator(*args, **kwargs):
            async with PidFile(path):
                return await func(*args, **kwargs)
        return decorator
    return wrapper
