# encoding: UTF-8

import json
import os
import sys

import click
import jinja2

from ws_properties.conversion.service import StandardConversionService
from ws_properties.environ.environment import StandardEnvironment
from ws_properties.environ.mapper import Mapper
from ws_properties.environ.mapper import ValueMapper
from ws_properties.environ.properties import DictPropertySource
from ws_properties.utils.imports import import_module_attr
from ws_properties.utils.j2 import WSPropertiesEnvironment
from ws_properties.utils.j2 import try_load_stdin


@click.group()
@click.option(
    '-p',
    '--profile',
    multiple=True,
    help='Additional profiles to load.'
)
@click.pass_context
def main(ctx, profile):
    """Universal utility to render settings"""

    environment = StandardEnvironment()
    environment.conversion_service = StandardConversionService()
    environment.profiles = profile
    environment.activate_profiles()

    ctx.obj = environment


@main.command()
@click.pass_obj
def profiles(environment):
    """Returns active profiles"""

    for profile in environment.profiles:
        click.echo(profile)


@main.command()
@click.argument('PATH')
@click.option(
    '-m',
    '--mapper',
    help='Conversion to apply on value.'
)
@click.option(
    '-d',
    '--default',
    help='Default property value.'
)
@click.option(
    '--to-json',
    is_flag=True,
    help='Conversion to apply on value.'
)
@click.pass_obj
def property(environment, path, mapper, default, to_json):
    """Returns single property"""

    if default is not None:
        environment.property_sources.append(
            DictPropertySource({
                path: default,
            })
        )

    if mapper:
        mapper = import_module_attr(mapper)
        is_exit_mapper = False
        if mapper is None:
            raise ValueError('Mapper not found')
        elif mapper is exit:
            is_exit_mapper = True
            mapper = ValueMapper(bool)
        elif isinstance(mapper, type):
            mapper = ValueMapper(mapper)
        elif not isinstance(mapper, Mapper):
            raise ValueError('Invalid mapper type')

        value = mapper.map(environment, path)
        if is_exit_mapper:
            exit(0 if value else 1)
    else:
        value = environment.get_property(path)

    if value is None:
        exit(1)

    if to_json:
        click.echo(json.dumps(value))
    else:
        click.echo(str(value))


@main.command()
@click.argument('STRING')
@click.option(
    '-i',
    '--ignore-unresolvable',
    is_flag=True,
    help='Ignore unresolvable placeholders (keep as is).'
)
@click.pass_obj
def resolve(settings, string, ignore_unresolvable):
    """Returns single property"""

    result = settings.resolve_placeholders(string, ignore_unresolvable)
    click.echo(result)


@main.command()
@click.option(
    '-s',
    '--source',
    required=False,
    default='-',
    type=click.Path(exists=True, dir_okay=False, allow_dash=True)
)
@click.option(
    '-d',
    '--destination',
    required=False,
    default='-',
    type=click.Path(dir_okay=False, writable=True, allow_dash=True)
)
@click.pass_obj
def template(environment, source, destination):
    """Render Jinja2 template"""

    j2 = WSPropertiesEnvironment(
        environment=environment,
        undefined=jinja2.StrictUndefined,
        trim_blocks=True,
        lstrip_blocks=True,
        keep_trailing_newline=True,
        loader=jinja2.ChoiceLoader([
            jinja2.FunctionLoader(try_load_stdin),
            jinja2.FileSystemLoader(os.getcwd()),
            jinja2.FileSystemLoader('/'),
        ]),
    )

    template = j2.get_template(source)
    out_stream = open(destination, 'wb') if destination != '-' else sys.stdout
    with out_stream:
        for chunk in template.generate():
            out_stream.write(chunk)


if __name__ == '__main__':
    main()
