# encoding: UTF-8

import abc
import datetime

import enum
import isodate


class ConversionError(ValueError):
    @staticmethod
    def create_default(source, target_type):
        return ConversionError(
            'Can not convert %r to %r' % (source, target_type)
        )


class Converter(object):
    __metaclass__ = abc.ABCMeta

    @abc.abstractmethod
    def convert(self, source, target_type, **hints):
        """
        Converts source to target type.
        Conversion may be tuned by providing hints.

        :param source: Source value to be converted
        :type source: Any

        :param target_type: Target type source will be converted to
        :type target_type: type

        :returns: Converted value or None if source was None.
        :rtype: Any

        :raises: ConversionError if source can not be converted to target type.
        """


class AbstractConverter(Converter):
    @abc.abstractmethod
    def _can_convert(self, source_type, target_type):
        """
        Check types before conversion.

        :param source_type: Type to be converted from
        :type source_type: type

        :param target_type: Type to be converted to
        :type target_type: type

        :returns: True on successful check
        :rtype: bool
        """

    @abc.abstractmethod
    def _do_convert(self, source, target_type, **hints):
        """
        Make conversion after successful type checking.
        Conversion may be tuned by providing hints.

        :param source: Source value to be converted
        :type source: Any

        :param target_type: Target type source will be converted to
        :type target_type: type

        :returns: Converted value or None if source was None.
        :rtype: Any

        :raises: ConversionError if source can not be converted to target type.
        """

    def convert(self, source, target_type, **hints):
        if source is None:
            return None
        elif self._can_convert(type(source), target_type):
            return self._do_convert(source, target_type, **hints)
        else:
            raise ConversionError.create_default(source, target_type)


class ToDateTimeConverter(AbstractConverter):
    def _can_convert(self, source_type, target_type):
        return (
                issubclass(source_type, (str, unicode)) and
                target_type is datetime.datetime
        )

    def _do_convert(self, source, target_type, **hints):
        try:
            return isodate.parse_datetime(source)
        except isodate.ISO8601Error:
            raise ConversionError.create_default(source, target_type)


class ToDateConverter(AbstractConverter):
    def _can_convert(self, source_type, target_type):
        return (
                issubclass(source_type, (str, unicode)) and
                target_type is datetime.date
        )

    def _do_convert(self, source, target_type, **hints):
        try:
            return isodate.parse_date(source)
        except isodate.ISO8601Error:
            raise ConversionError.create_default(source, target_type)


class ToTimeConverter(AbstractConverter):
    def _can_convert(self, source_type, target_type):
        return (
                issubclass(source_type, (str, unicode)) and
                target_type is datetime.time
        )

    def _do_convert(self, source, target_type, **hints):
        try:
            return isodate.parse_time(source)
        except isodate.ISO8601Error:
            raise ConversionError.create_default(source, target_type)


class ToTimeDeltaConverter(AbstractConverter):
    def _can_convert(self, source_type, target_type):
        return (
                issubclass(source_type, (str, unicode)) and
                target_type is datetime.timedelta
        )

    def _do_convert(self, source, target_type, **hints):
        try:
            return isodate.parse_duration(source)
        except isodate.ISO8601Error:
            raise ConversionError.create_default(source, target_type)


class ToBoolConverter(AbstractConverter):
    def _can_convert(self, source_type, target_type):
        return target_type is bool

    def _do_convert(self, source, target_type, **hints):
        if isinstance(source, (str, unicode)):
            try:
                return bool(int(source))
            except ValueError:
                lower_source = source.lower()
                if lower_source == 'true':
                    return True
                elif lower_source == 'false':
                    return False
                else:
                    raise ConversionError.create_default(source, target_type)
        elif isinstance(source, (int, long)):
            return bool(source)
        else:
            raise ConversionError.create_default(source, target_type)


class ToIntegerConverter(AbstractConverter):
    def _can_convert(self, source_type, target_type):
        return target_type is int or target_type is long

    def _do_convert(self, source, target_type, **hints):
        try:
            try:
                return target_type(source, **hints)
            except TypeError:
                return target_type(source)
        except ValueError:
            raise ConversionError.create_default(source, target_type)


class ToFloatConverter(AbstractConverter):
    def _can_convert(self, source_type, target_type):
        return target_type is float

    def _do_convert(self, source, target_type, **hints):
        try:
            return target_type(source, **hints)
        except ValueError:
            raise ConversionError.create_default(source, target_type)


class ToEnumConverter(AbstractConverter):
    def _can_convert(self, source_type, target_type):
        return issubclass(target_type, enum.Enum)

    def _do_convert(self, source, target_type, **hints):
        if hints.get('by_name', False):
            try:
                return getattr(target_type, source)
            except (AttributeError, TypeError):
                raise ConversionError.create_default(source, target_type)
        else:
            try:
                return target_type(source)
            except ValueError:
                raise ConversionError.create_default(source, target_type)


class CompositeConverter(Converter):
    def __init__(self):
        self.converters = []

    def convert(self, source, target_type, **hints):
        if source is None:
            return None
        elif type(source) == target_type:
            return source
        else:
            for converter in self.converters:
                try:
                    return converter.convert(source, target_type, **hints)
                except ConversionError:
                    pass

            raise ConversionError.create_default(source, target_type)
