# encoding: UTF-8

import itertools
import os
import sys

import orderedset

from ws_properties.conversion.converter import Converter
from ws_properties.conversion.service import ConversionService
from ws_properties.environ.properties import CompositePropertySource
from ws_properties.environ.properties import EnvironmentPropertySource
from ws_properties.environ.properties import PropertySource
from ws_properties.environ.properties import YAMLFilePropertySource


class Environment(PropertySource, Converter):
    PROFILES_ACTIVE_PROPERTY = 'application.profiles.active'

    def __init__(self):
        self.conversion_service = None  # type: ConversionService
        self.__properties = CompositePropertySource()
        self.__profiles = orderedset.OrderedSet()

        self._customize_property_sources(self.__properties.property_sources)

    def _customize_property_sources(self, property_sources):
        pass

    @property
    def property_sources(self):
        return self.__properties.property_sources

    def _get_active_profiles(self):
        if not self.__profiles:
            profiles = self.get_list_property(self.PROFILES_ACTIVE_PROPERTY)
            if profiles:
                self.profiles = profiles

        return self.__profiles

    @property
    def profiles(self):
        return list(self._get_active_profiles())

    @profiles.setter
    def profiles(self, profiles):
        self.__profiles.clear()
        for profile in profiles:
            self._validate_profile(profile)
            self.__profiles.add(profile)

    # noinspection PyMethodMayBeStatic
    def _validate_profile(self, profile):
        if not profile:
            raise ValueError('Invalid profile %r: must contains text' % profile)

    def add_profile(self, profile):
        self._validate_profile(profile)
        self._get_active_profiles()
        self.__profiles.add(profile)

    def has_prefix(self, prefix):
        return self.__properties.has_prefix(prefix)

    def get_property(self, name):
        value = self.__properties.get_property(name)
        if isinstance(value, (str, unicode)):
            value = self.resolve_placeholders(value)
        return value

    def get_list_property(self, name, delimiter=','):
        values = self.__properties.get_list_property(name, delimiter)
        if values is None:
            return None
        else:
            return map(self.resolve_placeholders, values)

    def resolve_placeholders(self, s, ignore_unresolvable=False):
        return self.__properties.resolve_placeholders(s, ignore_unresolvable)

    def convert(self, source, target_type, **hints):
        return self.conversion_service.convert(source, target_type, **hints)


class StandardEnvironment(Environment):
    CONFIG_NAME_PROPERTY = 'application.config.name'
    CONFIG_LOCATION_PROPERTY = 'application.config.location'
    CONFIG_ADD_LOCATION_PROPERTY = 'application.profiles.additional-paths'

    PROFILES_INCLUDE_PROPERTY = 'application.profiles.include'

    def __init__(self):
        super(StandardEnvironment, self).__init__()
        self.config_locations = []
        self.config_name = 'application'
        self.__profiles_activated = False

    def _customize_property_sources(self, property_sources):
        property_sources.append(EnvironmentPropertySource())

    def _get_default_config_location(self):
        return [
            os.path.join(sys.prefix, 'profiles') + '/',
            './',
            './profiles/',
        ]

    def _get_additional_config_locations(self):
        add_profiles_paths = self.get_list_property(
            self.CONFIG_ADD_LOCATION_PROPERTY,
        )
        return add_profiles_paths or []

    def _get_config_locations(self):
        config_location = self.get_list_property(
            self.CONFIG_LOCATION_PROPERTY,
        )

        if config_location is None:
            config_location = self._get_default_config_location()

        additional_config_location = self._get_additional_config_locations()
        config_location.extend(additional_config_location)

        return config_location

    def _get_include_profiles(self, property_source):
        value = property_source.get_property(self.PROFILES_INCLUDE_PROPERTY)
        value = value and value.strip()
        if value:
            value = self.resolve_placeholders(value)
            return map(str.strip, value.split(','))
        elif property_source.has_prefix(self.PROFILES_INCLUDE_PROPERTY + '['):
            values = []
            for i in itertools.count():
                ipath = self.PROFILES_INCLUDE_PROPERTY + '[' + str(i) + ']'
                value = property_source.get_property(ipath)
                value = value and self.resolve_placeholders(value)
                if value is None:
                    break
                else:
                    values.append(value)
            return values
        else:
            return []

    def _load_config(self, filename):
        result = CompositePropertySource()
        for config_location in reversed(self.config_locations):
            path = os.path.join(config_location, filename)
            if os.path.exists(path):
                result.property_sources.append(YAMLFilePropertySource(path))
        return result.property_sources and result or None

    def _load_recursive_configs(
            self,
            insert_position,
            profiles,
            _inprogress=set(),
            _loaded=orderedset.OrderedSet(),
    ):
        for profile in profiles:
            if profile in _inprogress:
                raise ValueError(
                    'Cyclic profile dependency found: %s.' % profile
                )

            _inprogress.add(profile)

            config_filename = self.config_name + '-' + profile + '.yml'
            config_source = self._load_config(config_filename)
            if config_source is not None:
                include_profiles = self._get_include_profiles(config_source)

                self._load_recursive_configs(
                    insert_position,
                    include_profiles,
                    _inprogress,
                    _loaded,
                )

                self.property_sources.insert(insert_position, config_source)

            _inprogress.discard(profile)
            profiles = self._get_active_profiles().copy()
            profiles.discard(profile)
            self.profiles = list(profiles) + [profile]

        return _loaded

    def activate_profiles(self):
        if self.__profiles_activated:
            raise RuntimeError('Profiles already activated')

        self.__profiles_activated = True

        self.config_locations = self._get_config_locations()
        config_name = self.get_property(self.CONFIG_NAME_PROPERTY)
        if config_name is not None:
            self.config_name = config_name

        config_source = self._load_config(self.config_name + '.yml')
        insert_position = len(self.property_sources)
        if config_source is not None:
            self.property_sources.append(config_source)

        self._load_recursive_configs(
            insert_position,
            self.profiles,
        )
