# encoding: UTF-8

import abc
import itertools

from ws_properties.conversion.converter import ConversionError
from ws_properties.environ.environment import Environment
from ws_properties.utils.imports import get_type
from ws_properties.utils.imports import import_module_attr


def get_mapper(value):
    if isinstance(value, (str, unicode)):
        value = import_module_attr(value)

    if isinstance(value, Mapper):
        return value
    elif isinstance(value, type):
        return ValueMapper(value)
    else:
        raise ValueError('Value %r can not be transformed to mapper' % value)


RaiseError = object()


class MissingError(ValueError):
    def __init__(self, message, name, *args):
        super(MissingError, self).__init__(message, name, *args)

    name = property(lambda self: self.args[1])


class Mapper(object):
    __metaclass__ = abc.ABCMeta

    @abc.abstractmethod
    def map(self, environment, path):
        raise NotImplementedError


class ValueMapper(Mapper):
    def __init__(
            self,
            target_type,
            default=RaiseError,
            ignore_unresolvable=False,
            **hints
    ):
        self.target_type = get_type(target_type)
        self.default = default
        self.ignore_unresolvable = ignore_unresolvable
        self.hints = hints

    def map(self, environment, path):
        # type: (Environment, ...) -> any
        value = environment.get_property(path)
        value = self.default if value is None else value

        if value is RaiseError:
            raise MissingError('Missing required property', path)
        else:
            return self.handle_raw_value(environment, value, path)

    def handle_raw_value(self, environment, value, path):
        if isinstance(value, str):
            value = environment.resolve_placeholders(
                value,
                self.ignore_unresolvable,
            )

        if self.target_type is not None:
            target_type = get_type(self.target_type)
            try:
                value = environment.convert(
                    value,
                    target_type,
                    **self.hints
                )
            except ConversionError as e:
                raise ValueError(e.message, path)

        return value


class ListMapper(Mapper):
    def __init__(self, mapper, min_occurrence=1, delimiter=','):
        self.mapper = get_mapper(mapper)
        self.min_occurrence = min_occurrence
        self.delimiter = delimiter

    def map(self, environment, path=''):
        if isinstance(self.mapper, ValueMapper):
            values = environment.get_list_property(path, self.delimiter)
            values = [] if values is None else [
                self.mapper.handle_raw_value(
                    environment,
                    value,
                    path + '[' + str(i) + ']',
                )
                for i, value in enumerate(values)
            ]
            if len(values) < self.min_occurrence:
                ipath = path + '[' + str(len(values)) + ']'
                raise MissingError('Missing required property', ipath)
            else:
                return values
        else:
            values = []
            for i in itertools.count():
                ipath = path + '[' + str(i) + ']'

                if (
                        i >= self.min_occurrence and
                        not environment.has_prefix(ipath)
                ):
                    break

                value = self.mapper.map(environment, ipath)
                values.append(value)
            return values or None


class ObjectMapper(Mapper):
    def __init__(self, **properties):
        self.properties = {k: get_mapper(v) for k, v in properties.items()}

    def map(self, environment, path=''):
        values = {}

        for name, property in self.properties.items():
            ppath = path + '.' + name if path else name
            value = property.map(environment, ppath)
            values[name] = value

        return values
