# encoding: UTF-8

import abc
import collections
import itertools
import locale
import os
import urllib2

import ruamel.yaml


class PlaceholderResolver(object):
    def __init__(
            self,
            property_source,
            prefix='${',
            suffix='}',
            value_sep=':',
            ignore_unresolvable=False
    ):
        super(PlaceholderResolver, self).__init__()
        self.property_source = property_source
        self.prefix = prefix
        self.suffix = suffix
        self.value_sep = value_sep
        self.ignore_unresolvable = ignore_unresolvable

    def resolve_placeholders(self, s):
        return self._parse_string_value(s, set())

    def _parse_string_value(self, s, visited):
        result = type(s)(s)
        start = result.find(self.prefix)

        while start != -1:
            end = self._find_placeholder_end(result, start)
            if end != -1:
                original = holder = result[start + len(self.prefix):end]
                if original in visited:
                    msg = 'Circular placeholder reference \'%s\'' % original
                    raise ValueError(msg)
                else:
                    visited.add(original)

                holder = self._parse_string_value(holder, visited)
                value = self.property_source.get_property(holder)

                if value is None and self.value_sep is not None:
                    sep = holder.find(self.value_sep)
                    if sep != -1:
                        actual = holder[:sep]
                        default = holder[sep + len(self.value_sep):]
                        value = self.property_source.get_property(actual)
                        if value is None:
                            value = default

                if value is not None:
                    value = self._parse_string_value(value, visited)
                    result = (result[:start] +
                              value +
                              result[end + len(self.suffix):])
                    start = result.find(self.prefix, start + len(value))
                elif self.ignore_unresolvable:
                    start = result.find(self.prefix, end + len(self.suffix))
                else:
                    msg = 'Could not resolve placeholder ' \
                          '\'%s\' in value \'%s\'' % (holder, s)
                    raise ValueError(msg)
                visited.remove(original)
            else:
                start -= 1

        return result

    def _find_placeholder_end(self, s, start):
        offset = start + len(self.prefix)
        within_nested = 0

        while offset < len(s):
            if s[offset:offset + len(self.suffix)] == self.suffix:
                if within_nested > 0:
                    within_nested -= 1
                    offset += len(self.suffix)
                else:
                    return offset
            elif s[offset:offset + len(self.prefix)] == self.prefix:
                within_nested += 1
                offset += len(self.prefix)
            else:
                offset += 1

        return -1


class PropertySource(object):
    __metaclass__ = abc.ABCMeta

    @abc.abstractmethod
    def has_prefix(self, prefix):
        raise NotImplementedError

    @abc.abstractmethod
    def get_property(self, name):
        raise NotImplementedError

    def get_list_property(self, name, delimiter=','):
        value = self.get_property(name)
        if value is None:
            values = []
            for i in itertools.count():
                value = self.get_property(name + '[' + str(i) + ']')
                if value is None:
                    break
                else:
                    values.append(value)
            return values or None
        elif value:
            return value.split(delimiter)
        else:
            return None

    def resolve_placeholders(self, s, ignore_unresolvable=False):
        resolver = PlaceholderResolver(
            property_source=self,
            ignore_unresolvable=ignore_unresolvable,
        )
        return resolver.resolve_placeholders(s)


class AbstractPropertySource(PropertySource):
    @abc.abstractmethod
    def _get_raw_property(self, name):
        raise NotImplementedError

    # noinspection PyMethodMayBeStatic
    def _encode_text(self, value):
        return value.encode(locale.getpreferredencoding())

    def _convert_to_string(self, value):
        if isinstance(value, bytes):
            return value
        elif isinstance(value, unicode):
            return self._encode_text(value)
        else:
            return str(value)

    def get_property(self, name):
        value = self._get_raw_property(name)
        if value is None:
            return None
        else:
            return self._convert_to_string(value)

    @staticmethod
    def _flatten(value, prefix='', ignore_duplicates=False):
        queue = collections.deque()
        queue.append((prefix, value))

        result = {}

        while queue:
            path, item = queue.popleft()

            if isinstance(item, collections.Mapping):
                queue.extend(
                    (path + ('' if item is value else '.') + k, v)
                    for k, v in item.items()
                )
            elif (not isinstance(item, (str, unicode)) and
                  isinstance(item, collections.Iterable)):
                queue.extend(
                    (path + '[' + str(k) + ']', v)
                    for k, v in enumerate(item)
                )
            else:
                if path in result and not ignore_duplicates:
                    raise ValueError(
                        'Produced path \'%s\' duplicates another one' % path
                    )
                else:
                    result[path] = item

        return result


class DictPropertySource(AbstractPropertySource):
    def __init__(self, value, prefix=''):
        self._data = self._flatten(value, prefix)

    def has_prefix(self, prefix):
        for k in self._data:
            if k.startswith(prefix):
                return True
        return False

    def _get_raw_property(self, name):
        return self._data.get(name)

    def __repr__(self):
        return '<%s(%d)>' % (
            self.__class__.__name__,
            len(self._data),
        )


class EnvironmentPropertySource(AbstractPropertySource):
    def has_prefix(self, prefix):
        normalized_prefix = prefix.upper().replace('.', '_')
        for k in os.environ:
            if k.startswith(normalized_prefix):
                return True
        return False

    def _get_raw_property(self, name):
        normalized_name = name.upper().replace('.', '_')
        return os.environ.get(normalized_name)

    def __repr__(self):
        return '<%s(%d)>' % (
            self.__class__.__name__,
            len(os.environ),
        )


class YAMLPropertySourceLoader(ruamel.yaml.SafeLoader):
    _RESTRICTED_TAGS = {
        'tag:yaml.org,2002:bool',
        'tag:yaml.org,2002:int',
        'tag:yaml.org,2002:float',
        'tag:yaml.org,2002:timestamp',
        'tag:yaml.org,2002:null',
    }

    def add_version_implicit_resolver(self, version, tag, regexp, first):
        if tag in self._RESTRICTED_TAGS:
            return

        super(YAMLPropertySourceLoader, self).add_version_implicit_resolver(
            version,
            tag,
            regexp,
            first,
        )


class YAMLStreamPropertySource(DictPropertySource):
    def __init__(self, stream, prefix=''):
        self._loader = YAMLPropertySourceLoader(stream)
        super(YAMLStreamPropertySource, self).__init__(
            value=self._loader.get_single_data(),
            prefix=prefix,
        )
        self._stream_repr = repr(stream)

    def _encode_text(self, value):
        encoding = self._loader.encoding or locale.getpreferredencoding()
        return value.encode(encoding)

    def __repr__(self):
        return '<%s(%r)>' % (
            self.__class__.__name__,
            self._stream_repr,
        )


class YAMLFilePropertySource(YAMLStreamPropertySource):
    def __init__(self, filename):
        with open(filename, 'r') as f:
            super(YAMLFilePropertySource, self).__init__(f)
        self.filename = filename

    def __repr__(self):
        return '<%s(%r)>' % (
            self.__class__.__name__,
            self.filename,
        )


class YAMLUrlPropertySource(YAMLStreamPropertySource):
    def __init__(self, url):
        resp = urllib2.urlopen(url)
        super(YAMLUrlPropertySource, self).__init__(resp)
        self.url = url

    def __repr__(self):
        return '<%s(%r)>' % (
            self.__class__.__name__,
            self.url,
        )


class CompositePropertySource(PropertySource):
    def __init__(self):
        super(CompositePropertySource, self).__init__()
        self.property_sources = []

    def has_prefix(self, prefix):
        for property_source in self.property_sources:
            if property_source.has_prefix(prefix):
                return True
        return False

    def get_property(self, name):
        for property_source in self.property_sources:
            value = property_source.get_property(name)
            if value is not None:
                return value

        return None

    def get_list_property(self, name, delimiter=','):
        for property_source in self.property_sources:
            value = property_source.get_list_property(name, delimiter)
            if value is not None:
                return value

        return None

    def __repr__(self):
        return '<%s(%r)>' % (
            self.__class__.__name__,
            self.property_sources,
        )
