# encoding: UTF-8

import subprocess
import sys

import jinja2

from ws_properties.environ.mapper import ListMapper, RaiseError, MissingError
from ws_properties.environ.mapper import ObjectMapper
from ws_properties.environ.mapper import ValueMapper
from ws_properties.environ.mapper import get_mapper
from ws_properties.utils.imports import get_type


def filter_escape(value, quote_char='"'):
    return value.replace('\\', '\\\\').replace(quote_char, '\\' + quote_char)


class FilterConvert(object):
    environmentfilter = True

    def __init__(self, environment):
        self.environnment = environment

    def __call__(self, env, value, target_type, **hints):
        target_type = get_type(target_type)
        return self.environnment.convert(value, target_type, **hints)


def helper_system(value):
    proc = subprocess.Popen(value, shell=True, stdout=subprocess.PIPE)
    stdout, _ = proc.communicate()
    if proc.returncode != 0:
        msg = 'Command \'%s\' failed with exit code %d'
        raise RuntimeError(msg % (value, proc.returncode))

    return [
        line.strip()
        for line in stdout.splitlines()
    ]


class HelperProperty(object):
    def __init__(self, environment, undefined):
        self.environment = environment
        self.undefined = undefined

    def __call__(self, path, target_type=str):
        mapper = ValueMapper(target_type)
        try:
            return mapper.map(self.environment, path)
        except MissingError:
            return self.undefined(name=path)


class HelperListProperty(object):
    def __init__(self, environment, undefined):
        self.environment = environment
        self.undefined = undefined

    def __call__(self, path, target_type=str, delimiter=','):
        mapper = ListMapper(
            ValueMapper(target_type),
            min_occurrence=0,
            delimiter=delimiter,
        )
        return mapper.map(self.environment, path) or self.undefined(name=path)


class HelperMapProperty(object):
    def __init__(self, environment):
        self.environment = environment

    def __call__(self, path, mapper):
        mapper = get_mapper(mapper)
        return mapper.map(self.environment, path)


def try_load_stdin(name):
    if name == '-':
        return sys.stdin.read()


class WSPropertiesEnvironment(jinja2.Environment):
    def __init__(self, environment, *args, **kwargs):
        super(WSPropertiesEnvironment, self).__init__(
            *args,
            **kwargs
        )
        self.filters.update({
            'e': filter_escape,
            'escape': filter_escape,
            'convert': FilterConvert(environment),
        })
        self.globals.update({
            'system': helper_system,
            'prop': HelperProperty(environment, self.undefined),
            'lprop': HelperListProperty(environment, self.undefined),
            'mprop': HelperMapProperty(environment),
            'ValueMapper': ValueMapper,
            'ListMapper': ListMapper,
            'ObjectMapper': ObjectMapper,
            'RaiseError': RaiseError,
        })
