package ru.yandex.chemodan.app.ydbtest;

import java.time.Instant;
import java.util.concurrent.ExecutionException;

import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.DataQuery;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.rpc.grpc.GrpcSchemeRpc;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;
import com.yandex.ydb.table.transaction.Transaction;
import com.yandex.ydb.table.transaction.TransactionMode;
import com.yandex.ydb.table.transaction.TxControl;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructType;
import com.yandex.ydb.table.values.Value;
import org.junit.Ignore;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.test.Assert;

/**
 * @author tolmalev
 */
@Ignore
public class YdbTest {
    private TableClient tableClient;
    private SchemeClient schemeClient;

    private final String database;
    private final String endpoint;

    private static StructType DATABASE_STRUCT_TYPE = StructType.of(
            Cf.list("user_id", "app", "dbId", "handle", "rev", "creation_time", "modification_time", "size", "records_count", "description"),
            Cf.list(
                    PrimitiveType.string(),
                    PrimitiveType.string(),
                    PrimitiveType.string(),
                    PrimitiveType.string(),
                    PrimitiveType.int64(),
                    PrimitiveType.timestamp(),
                    PrimitiveType.timestamp(),
                    PrimitiveType.int64(),
                    PrimitiveType.int64(),
                    PrimitiveType.string()
            )
    );

    public YdbTest() throws ExecutionException, InterruptedException {
        endpoint = new File2("ydb_endpoint.txt").readFirstLine();
        database = new File2("ydb_database.txt").readFirstLine();

        RpcTransport transport = GrpcTransport.forEndpoint(endpoint, database).build();

        tableClient = TableClient.newClient(GrpcTableRpc.ownTransport(transport))
                .sessionPoolSize(10, 100)
                .build();

        schemeClient = SchemeClient.newClient(GrpcSchemeRpc.useTransport(transport))
                .build();

        createTables();
    }

    public void createTables() throws ExecutionException, InterruptedException {
        Session session = getSession();

        try {
            schemeClient.listDirectory(database).get().expect("").getChildren().forEach(entty -> {
                try {
                    session.dropTable(database + "/" + entty.getName()).get().expect("Failed to drop table");
                } catch (Exception e) {
                    throw ExceptionUtils.translate(e);
                }
            });

            session
                    .createTable(database + "/databases", TableDescription.newBuilder()
                            .addNullableColumn("user_id", PrimitiveType.string())
                            .addNullableColumn("app", PrimitiveType.string())
                            .addNullableColumn("dbId", PrimitiveType.string())
                            .addNullableColumn("handle", PrimitiveType.string())
                            .addNullableColumn("rev", PrimitiveType.int64())
                            .addNullableColumn("creation_time", PrimitiveType.timestamp())
                            .addNullableColumn("modification_time", PrimitiveType.timestamp())
                            .addNullableColumn("size", PrimitiveType.int64())
                            .addNullableColumn("records_count", PrimitiveType.int64())
                            .addNullableColumn("description", PrimitiveType.string())
                            .setPrimaryKeys("user_id", "app", "dbId")
                            .build())
                    .get()
                    .expect("Failed to create table");
        } finally {
            session.release();
        }
    }

    private Session getSession() throws InterruptedException, ExecutionException {
        return tableClient.createSession().get().expect("Can't create session");
    }

    @Test
    public void bulkInsert() throws ExecutionException, InterruptedException {
        Session session = getSession();

        ListType listType = ListType.of(DATABASE_STRUCT_TYPE);
        String sql = "" +
                "DECLARE $items AS \"" + listType + "\";\n" +
                "INSERT INTO databases SELECT * FROM AS_TABLE($items);";

        System.out.println(sql);

        DataQuery dataQuery = session.prepareDataQuery(sql).get().expect("");
        Params params = dataQuery.newParams()
                .put("$items", listType.newValue(Cf.range(0, 10).map(i -> {

                    MapF<String, Value> values = Tuple2List.<String, Value>fromPairs(
                            "user_id", PrimitiveValue.string(("uid_" + i).getBytes()),
                            "app", PrimitiveValue.string(("app_" + i).getBytes()),
                            "dbId", PrimitiveValue.string(("dbId_" + i).getBytes()),
                            "handle", PrimitiveValue.string(("handle_" + i).getBytes()),
                            "rev", PrimitiveValue.int64(i),
                            "creation_time", PrimitiveValue.timestamp(Instant.ofEpochMilli(i * 1000000)),
                            "modification_time", PrimitiveValue.timestamp(Instant.ofEpochMilli(i * 1000000)),
                            "size", PrimitiveValue.int64(DataSize.fromMegaBytes(i).toBytes()),
                            "records_count", PrimitiveValue.int64(i),
                            "description", PrimitiveValue.string(("description_" + i).getBytes())
                    ).toMap();

                    return DATABASE_STRUCT_TYPE.newValue(values);
                })));

        dataQuery.execute(TxControl.serializableRw(), params).get().expect("Failed to execute query");

        session.release();
    }

    @Test
    public void concurrentUpdates() throws ExecutionException, InterruptedException {
        //prepare date
        bulkInsert();

        Session s1 = getSession();
        Session s2 = getSession();

        Transaction t1 = s1.beginTransaction(TransactionMode.SERIALIZABLE_READ_WRITE).get().expect("");
        Transaction t2 = s2.beginTransaction(TransactionMode.SERIALIZABLE_READ_WRITE).get().expect("");

        s1.executeDataQuery("UPDATE databases SET rev = 100 WHERE user_id = 'uid_1'", TxControl.id(t1).setCommitTx(false)).get().expect("");
        s2.executeDataQuery("UPDATE databases SET rev = 200 WHERE user_id = 'uid_1'", TxControl.id(t2).setCommitTx(false)).get().expect("");

        t1.commit().get().expect("");
        Status status = t2.commit().get();

        Assert.isFalse(status.isSuccess());
        Assert.equals(StatusCode.ABORTED, status.getCode());
    }

    @Test
    public void concurrentUpdateAndRead() throws ExecutionException, InterruptedException {
        //prepare date
        bulkInsert();

        Session s1 = getSession();
        Session s2 = getSession();

        Transaction t1 = s1.beginTransaction(TransactionMode.SERIALIZABLE_READ_WRITE).get().expect("");
        Transaction t2 = s2.beginTransaction(TransactionMode.SERIALIZABLE_READ_WRITE).get().expect("");

        s1.executeDataQuery("UPDATE databases SET rev = 100 WHERE user_id = 'uid_1'", TxControl.id(t1).setCommitTx(false)).get().expect("");
        s2.executeDataQuery("SELECT * FROM databases WHERE user_id = 'uid_1'", TxControl.id(t2).setCommitTx(false)).get().expect("");

        t1.commit().get().expect("");
        Status status = t2.commit().get();

        Assert.isFalse(status.isSuccess());
        Assert.equals(StatusCode.ABORTED, status.getCode());
    }

    @Test
    public void concurrentReadAndUpdate() throws ExecutionException, InterruptedException {
        //prepare date
        bulkInsert();

        Session s1 = getSession();
        Session s2 = getSession();

        Transaction t1 = s1.beginTransaction(TransactionMode.SERIALIZABLE_READ_WRITE).get().expect("");
        Transaction t2 = s2.beginTransaction(TransactionMode.SERIALIZABLE_READ_WRITE).get().expect("");

        s1.executeDataQuery("UPDATE databases SET rev = 100 WHERE user_id = 'uid_1'", TxControl.id(t1).setCommitTx(false)).get().expect("");
        s2.executeDataQuery("SELECT * FROM databases WHERE user_id = 'uid_1'", TxControl.id(t2).setCommitTx(false)).get().expect("");

        t2.commit().get().expect("");
        t1.commit().get().expect("");
    }
}
