# -*- coding: utf-8 -*-

from logging import getLogger
from os import environ, devnull, makedirs, remove
from os.path import join, exists
from time import sleep
from subprocess import Popen, TimeoutExpired
from signal import SIGINT
from contextlib import suppress
from shutil import rmtree
from random import random
from drive.config import ANIMALS


_log = getLogger(__name__)

_null_fd = open(devnull, "w")


def _wait(proc):
    try:
        proc.wait(timeout=8)
    except TimeoutExpired:
        proc.kill()
        proc.wait(timeout=8)


class Display(object):
    def __init__(self, display, vnc_port=None):
        self._display = display
        self._xvfb = None
        self._fluxbox = None
        self._x11vnc = None
        self._ffmpeg = None
        self._vnc_port = vnc_port

    def open(self):
        if self._display != ":0":
            _log.info("Open display: %s", self._display)
            self._xvfb = Popen(
                ("Xvfb", self._display, "-screen", "0", "1366x768x24"),
                stdout=_null_fd, stderr=_null_fd, env=self._get_environ(),
            )
            sleep(8)
            self._fluxbox = Popen(
                ("fluxbox",),
                stdout=_null_fd, stderr=_null_fd, env=self._get_environ(),
            )
            sleep(8)
        if self._vnc_port:
            self._x11vnc = Popen(
                (
                    "x11vnc", "-forever",
                    "-display", self._display,
                    "-rfbport", str(self._vnc_port),
                ),
                stdout=_null_fd, stderr=_null_fd, env=self._get_environ(),
            )
            sleep(1)

    def _get_environ(self):
        env = environ.copy()
        env["DISPLAY"] = self._display
        return env

    def close(self):
        _log.info("Close display: %s", self._display)
        if self._x11vnc:
            self._x11vnc.send_signal(SIGINT)
        if self._fluxbox:
            self._fluxbox.send_signal(SIGINT)
        if self._xvfb:
            self._xvfb.send_signal(SIGINT)
        if self._x11vnc:
            _wait(self._x11vnc)
        if self._fluxbox:
            _wait(self._fluxbox)
        if self._xvfb:
            _wait(self._xvfb)
        self._x11vnc = self._fluxbox = self._xvfb = None

    def input(self, string):
        self._xte("str {}".format(string))

    def slow_input(self, string):
        for char in string:
            sleep(0.1 + 0.2 * random())
            self.input(char)

    def press(self, key, mods=()):
        for mod in mods:
            self._xte("keydown {}".format(mod))
        self._xte("key {}".format(key))
        for mod in mods[::-1]:
            self._xte("keyup {}".format(mod))

    def press_tab(self):
        self.press("Tab")

    def press_space(self):
        self.press("space")

    def press_delete(self):
        self.press("Delete")

    def press_return(self):
        self.press("Return")

    def press_home(self):
        self.press("Home")

    def press_backspace(self):
        self.press("BackSpace")

    def press_escape(self):
        self.press("Escape")

    def _xte(self, *commands):
        _wait(Popen(
            ["xte"] + list(commands),
            stdout=_null_fd, stderr=_null_fd, env=self._get_environ(),
        ))

    def __enter__(self):
        self.open()
        return self

    def __exit__(self, type, value, traceback):
        self.close()


class Recorder(object):
    def __init__(self, display, record_file=None):
        self._display = display
        self._record_file = record_file
        self._ffmpeg = None

    def open(self):
        if self._record_file:
            _log.info(
                "Start recording: {}".format(self._record_file),
            )
            self._ffmpeg = Popen(
                (
                    "ffmpeg", "-y", "-f", "x11grab",
                    "-video_size", "1366x768",
                    "-framerate", "15",
                    "-i", self._display._display,
                    # "-async", "1",
                    "-vsync", "1",
                    "-preset", "ultrafast",
                    self._record_file,
                ),
                stdout=_null_fd, stderr=_null_fd,
                env=self._display._get_environ(),
            )
            sleep(3)

    def close(self):
        if self._ffmpeg:
            _log.info("Stop recording")
            self._ffmpeg.send_signal(SIGINT)
            _wait(self._ffmpeg)
            self._ffmpeg = None

    def __enter__(self):
        self.open()
        return self

    def __exit__(self, type, value, traceback):
        self.close()


_USER_JS = """
user_pref("network.proxy.http", "animals.search.yandex.net");
user_pref("network.proxy.http_port", 4004);
user_pref("network.proxy.ssl", "animals.search.yandex.net");
user_pref("network.proxy.ssl_port", 4004);
user_pref("network.proxy.type", 1);
"""


class Firefox(object):
    def __init__(self, display, profile, use_proxy=True):
        self._display = display
        self._profile = profile
        self._proc = None
        self._use_proxy = use_proxy

    def open(self):
        self._setup_profile()
        _log.info("Start Firefox")
        self._proc = Popen(
            ("firefox", "--profile", self._profile),
            stdout=_null_fd, stderr=_null_fd,
            env=self._display._get_environ(),
        )
        sleep(15)
        if self._use_proxy:
            self._setup_proxy()

    def _setup_profile(self):
        _log.info("Prepare Firefox")
        makedirs(self._profile, exist_ok=True)
        if self._use_proxy:
            with open(join(self._profile, "user.js"), "w") as fd:
                fd.write(_USER_JS)

    def _setup_proxy(self):
        _log.info("Setup proxy")
        self._try_setup_proxy()
        sleep(8)
        self.open_new_tab("example.com")
        sleep(8)
        self._try_setup_proxy()
        self.close_tab()

    def open_new_tab(self, url=None):
        _log.info("Open Firefox tab: %s", url)
        self._display.press("t", ("Control_L",))
        if url:
            sleep(0.1)
            self._display.input(url)
            self._display.press_delete()
            self._display.press_return()

    def close_tab(self):
        self._display.press("w", ["Control_L"])
        sleep(0.1)

    def _try_setup_proxy(self):
        _log.info("Trying setup proxy")
        self._display.input(ANIMALS["auth"]["login"])
        sleep(1)
        self._display.press_tab()
        sleep(1)
        self._display.input(ANIMALS["auth"]["token"])
        sleep(1)
        self._display.press_tab()
        sleep(1)
        self._display.press_tab()
        sleep(1)
        self._display.press_tab()
        sleep(1)
        self._display.press_space()
        sleep(1)

    def save_html(self, file_path):
        self._display.press("s", ["Control_L"])
        sleep(5)
        self._display.press("a", ["Control_L"])
        self._display.press_backspace()
        if not file_path.endswith(".html"):
            file_path += ".html"
        with suppress(FileNotFoundError):
            remove(file_path)
        self._display.input(file_path)
        sleep(0.1)
        self._display.press_return()
        for _ in range(1, 100):
            sleep(0.1)
            if exists(file_path):
                sleep(3)
                break
        with suppress(FileNotFoundError):
            rmtree(file_path[:-5] + "_files")
        self._display.press_escape()

    def close(self):
        _log.info("Stop Firefox")
        self._proc.send_signal(SIGINT)
        _wait(self._proc)

    def __enter__(self):
        self.open()
        return self

    def __exit__(self, type, value, traceback):
        self.close()
