# -*- coding: utf-8 -*-

from logging import getLogger
from os import environ, listdir, remove, mkdir
from os.path import join, exists
from time import sleep
from contextlib import contextmanager, suppress
from tempfile import mkdtemp
from shutil import rmtree
from random import random
from datetime import timedelta
from bs4 import BeautifulSoup
from threading import Thread, Event
from queue import Queue, Empty
from requests import post
from sqlalchemy import tablesample
from sqlalchemy.orm import aliased
from sqlalchemy.orm.exc import NoResultFound
from sqlalchemy.sql.expression import func
from yt.wrapper import TablePath
from drive.analytics.pybase.models import session
from drive.analytics.pybase.models.gibdd import LicenseCheck
from drive.analytics.pybase.config import TVM2, LICENSE_CHECKS
from drive.analytics.pybase.helpers import get_solomon, get_yt, yt_abspath
from drive.library.py.time import now, from_string, to_timestamp
from .gui import Display, Firefox, Recorder
# Deprecated.
from drive.utils.tvm2 import ServiceTicket


_log = getLogger(__name__)
PAGE_LOAD_WAIT = float(environ.get("PAGE_LOAD_WAIT", "20"))
SEND_FORM_WAIT = float(environ.get("SEND_FORM_WAIT", "20"))


@contextmanager
def temp_dir():
    path = mkdtemp()
    yield path
    rmtree(path)


def run_license_check(check, dir, display, firefox):
    _log.info("Parsing {}".format(check.license_number))
    result_file = join(dir, "result.html")
    firefox.open_new_tab("https://xn--90adear.xn--p1ai/check/driver")
    sleep(PAGE_LOAD_WAIT + 5 * random())
    display.slow_input(check.license_number)
    sleep(0.1 + 0.1 * random())
    display.press_tab()
    sleep(0.1 + 0.1 * random())
    display.press_home()
    sleep(1 + 1 * random())
    display.slow_input(check.license_issue_date.strftime("%d%m%y"))
    sleep(0.1 + 0.1 * random())
    display.press_tab()
    sleep(1 + 1 * random())
    _log.info("Send license check form")
    display.press_return()
    sleep(SEND_FORM_WAIT + 5 * random())
    display.press_escape()
    sleep(2 + random())
    firefox.save_html(result_file)
    firefox.close_tab()
    with open(result_file) as fd:
        raw_result = fd.read()
    return raw_result


def parse_date(date):
    if not date or date in ("00.00.0000", "нет данных"):
        return None
    for fmt in ("%d.%m.%Y", "%d%m%y", "%Y-%m-%d"):
        try:
            return from_string(date, fmt).strftime("%Y-%m-%d")
        except ValueError:
            continue
    raise ValueError(date)


def parse_period(period):
    if not period or period == "нет данных":
        return None
    if period.endswith("мес."):
        return dict(
            months=int(period[:-len("мес.")].strip()),
        )
    raise ValueError(period)


def parse_status(status):
    status = status.strip()
    if status in (
        "Окончено исчисление срока лишения права управления (возвращено ВУ)",
    ):
        return "finished"
    if status in (
        "Начато исчисление срока лишения права управления",
        "Постановление о лишении права управления ТС вступило в законную силу",
        "Поступление информации об уплате штрафа",
        "Проведение проверки знаний ПДД",
        "Вынесено постановление о лишении права управления ТС",
        "Поступление информации об уплате штрафа (от банка)",
    ):
        return "started"
    if status in (
        "Исчисление срока лишения права управления ТС прервано",
    ):
        return "paused"
    if status in (
        "нет данных",
    ):
        return "no_data"
    raise ValueError(status)


def parse_check_status(status):
    status = status.strip()
    if status == "Проверка завершилась ошибкой.":
        return "error"
    if status == "Проверка не запрашивалась":
        return "no_request"
    if status == "В результате проверки не были найдены сведения об указанном водительском удостоверении.":
        return "no_data"
    if status == "Проверка с помощью Google reCaptcha v3 не была пройдена, повторите попытку.":
        return "recaptcha"
    raise ValueError(status)


def parse_raw_result(raw):
    html = (
        BeautifulSoup(raw, features="lxml").
        find("div", {"id": "checkDriverContainer"})
    )
    if not html:
        raise RuntimeError("result does not contain check container")
    try:
        number = html.find("input", {"id": "checkDriverNum"}).get("value")
        if not number:
            raise RuntimeError("result does not contain license number")
        docum = html.find("ul", {"class": "fields-list doc"})
        if not docum:
            raise RuntimeError("result does not contain document info")
        docum_fields = docum.find_all("span", {"class": "field"})
        result = dict(
            status="success",
            number=number,
            birth_date=parse_date(docum_fields[0].text),
            issue_date=parse_date(docum_fields[1].text),
            expire_date=parse_date(docum_fields[2].text),
            deps=list(),
        )
        decis = html.find("ul", {"class": "fields-list decis"})
        if not decis:
            raise RuntimeError("result does not contain decis info")
        for item in decis.find_all("ul", {"class": "decis-item"}):
            vals = list()
            for field in item.find_all("li"):
                vals.append(field.find("span", {"class": "field"}).text)
            result["deps"].append(dict(
                birthplace=vals[0],
                start_date=parse_date(vals[1]),
                period=parse_period(vals[2]),
                status=parse_status(vals[3]),
                raw_status=vals[3].strip(),
            ))
        return result
    except RuntimeError:
        message = html.find("p", {"class": "check-space check-message"})
        if not message:
            raise
        return dict(
            status=parse_check_status(message.text),
            raw_status=message.text.strip(),
        )


class Worker(Thread):
    def __init__(self, closer, queue, display):
        super(Worker, self).__init__()
        self._closer = closer
        self._queue = queue
        self._display = display
        self._solomon = get_solomon("gibdd")
        self._tvm_cache = {}

    def run(self):
        while self._run_display():
            pass

    def _run_display(self):
        use_proxy = int(environ.get("USE_PROXY", 1))
        need_record = int(environ.get("RECORD", 0))
        with Display(self._display) as display:
            with temp_dir() as dir:
                firefox_dir = join(dir, "firefox")
                with Firefox(display, firefox_dir, use_proxy) as firefox:
                    for _ in range(8):
                        record_path = None
                        try:
                            check = self._queue.get(timeout=1)
                            if need_record:
                                record_path = "records/{}-{}.mp4".format(check.license_number, to_timestamp(now()))
                            with Recorder(display, record_path):
                                self._run_check(check, dir, display, firefox)
                            if need_record:
                                with suppress(FileNotFoundError):
                                    remove(record_path)
                        except Empty:
                            if self._closer.is_set():
                                _log.info("Worker stopped")
                                return False
                        except KeyboardInterrupt:
                            raise
                        except Exception as exc:
                            _log.exception(exc)
        return True

    def _parse_check(self, check, dir, display, firefox):
        raw_result = run_license_check(check, dir, display, firefox)
        self._solomon.signal("gibdd.license_check.run_success", 1)
        check.raw_result = raw_result
        check.result = parse_raw_result(raw_result)
        self._solomon.signal("gibdd.license_check.parse_success", 1)
        check.result_time = now()
        if check.result["status"] not in ("success", "no_data"):
            self._solomon.signal("gibdd.license_check.data_failure", 1)
            session.commit()
            raise RuntimeError("unable load driver license information")
        check.check_next_time = None
        check.notify_next_time = now()

    def _run_check(self, check, dir, display, firefox):
        self._solomon.signal("gibdd.license_check.run_start", 1)
        self._notify(check, "CheckStart", dict(
            CheckRetry=check.check_retry,
        ))
        try:
            session.add(check)
            last_exc = None
            for _ in range(5):
                try:
                    self._parse_check(check, dir, display, firefox)
                    last_exc = None
                    break
                except Exception as exc:
                    last_exc = exc
            if last_exc:
                raise last_exc
            session.commit()
            self._solomon.signal("gibdd.license_check.success", 1)
            self._notify(check, "CheckSuccess", dict(
                CheckRetry=check.check_retry,
            ))
        except Exception as exc:
            session.rollback()
            _log.exception(exc)
            self._solomon.signal("gibdd.license_check.failure", 1)
            self._notify(check, "CheckFailure", dict(
                CheckRetry=check.check_retry,
            ))
        finally:
            session.close()

    def _notify(self, check, type, data):
        if check.callback == "drive":
            return
        try:
            result = dict(
                Type=type,
                ID=check.id,
                Callback=check.callback,
                CallbackData=check.callback_data,
                Data=data,
            )
            callback = LICENSE_CHECKS["callbacks"][check.callback]
            if callback["target"] not in self._tvm_cache:
                self._tvm_cache[callback["target"]] = ServiceTicket(
                    base_url="https://tvm-api.yandex.net",
                    source=TVM2["source"],
                    target=callback["target"],
                    secret=str(TVM2["secret"]),
                )
            ticket = self._tvm_cache[callback["target"]]
            endpoint = callback["endpoint"]
            if check.callback_url:
                endpoint = check.callback_url
            resp = post(
                endpoint,
                headers={
                    "X-Ya-Service-Ticket": ticket.content,
                },
                json=result,
            )
            resp.raise_for_status()
            self._solomon.signal("gibdd.license_check.scraper_notify_success", 1)
        except Exception as exc:
            _log.exception(exc)
            self._solomon.signal("gibdd.license_check.scraper_notify_failure", 1)


def is_valid_license(s):
    for c in s:
        if c.isnumeric():
            continue
        if ord('A') <= ord(c) <= ord('Z'):
            continue
        if ord('a') <= ord(c) <= ord('z'):
            continue
        return False
    return True


def export_main():
    limit = 1000
    yt = get_yt()
    while True:
        checks = list(
            session.query(LicenseCheck).
            filter(LicenseCheck.check_next_time.is_(None)).
            filter(LicenseCheck.notify_next_time.is_(None)).
            order_by(LicenseCheck.id).limit(limit)
        )
        try:
            rows = []
            for check in checks:
                session.delete(check)
                rows.append(dict(
                    id=check.id,
                    license_number=check.license_number,
                    license_issue_date=check.license_issue_date.strftime("%Y-%m-%d")
                        if check.license_issue_date else None,
                    result=check.result,
                    raw_result=check.raw_result,
                    check_retry=check.check_retry,
                    check_tries=check.check_tries,
                    notify_retry=check.notify_retry,
                    callback=check.callback,
                    callback_data=check.callback_data,
                    result_time=check.result_time.strftime("%Y-%m-%d %H:%M:%S")
                        if check.result_time else None,
                    priority=check.priority,
                    callback_url=check.callback_url,
                ))
            yt.write_table(
                TablePath(yt_abspath(LICENSE_CHECKS["table"]), append=True),
                rows,
            )
            session.commit()
        except Exception as exc:
            session.rollback()
            raise exc
        finally:
            session.close()
        if len(checks) < limit:
            break


def cleanup():
    if not int(environ.get("CLEANUP", 1)):
        return
    for name in listdir("/tmp"):
        try:
            path = join("/tmp", name)
            if name.startswith("Temp-"):
                rmtree(path)
            elif name.startswith("tmpaddon-"):
                remove(path)
        except Exception as exc:
            _log.exception(exc)


def main():
    cleanup()
    parallel = int(environ.get("WORKERS", 8))
    checks_limit = int(environ.get("CHECKS_LIMIT", 100))
    display_base = int(environ.get("DISPLAY_BASE", 10))
    workers = list()
    closer = Event()
    queue = Queue(maxsize=parallel)
    if not exists("records"):
        mkdir("records")
    for i in range(parallel):
        worker = Worker(closer, queue, ":{}".format(display_base + i))
        workers.append(worker)
        worker.start()
    try:
        for _ in range(checks_limit):
            try:
                check_alias = aliased(LicenseCheck, tablesample(LicenseCheck, 2))
                check = (
                    session.query(check_alias).
                    filter(check_alias.check_next_time <= now()).
                    order_by((func.random() * (10 + func.log(check_alias.priority))).desc()).
                    limit(1).with_for_update().one()
                )
                if check.check_retry > check.check_tries:
                    check.check_next_time = None
                    check.notify_next_time = now()
                    session.commit()
                    continue
                check.check_retry += 1
                check.check_next_time = (
                    now() + timedelta(minutes=5 + check.check_retry)
                )
                session.commit()
            except NoResultFound:
                _log.info("No more checks in queue")
                break
            except Exception as exc:
                session.rollback()
                _log.exception(exc)
                continue
            if not is_valid_license(check.license_number):
                _log.info(
                    "License number is invalid: {}".
                    format(check.license_number),
                )
                try:
                    check.result = dict(
                        status="invalid",
                        raw_status="License number is invalid",
                    )
                    check.check_next_time = None
                    check.notify_next_time = now()
                    session.commit()
                except Exception as exc:
                    session.rollback()
                    _log.exception(exc)
                continue
            session.expunge(check)
            queue.put(check)
            _log.debug("Action put into the queue")
    except Exception as exc:
        _log.exception(exc)
        raise
    finally:
        _log.info("Stop workers")
        closer.set()
        for worker in workers:
            worker.join()
        _log.info("Workers joined")


if __name__ == "__main__":
    main()
