from json import dump
from contextlib import suppress


class Document:
    __documents = {}

    def __init_subclass__(cls):
        with suppress(NotImplementedError):
            Document.__documents[cls.get_name()] = cls

    @staticmethod
    def get_document(name):
        return Document.__documents[name]

    @staticmethod
    def get_documents():
        return list(Document.__documents.values())

    @classmethod
    def get_name(cls):
        raise NotImplementedError

    @classmethod
    def get_scheme(cls):
        raise NotImplementedError

    @classmethod
    def parse_document(cls, input_file, output_file, meta=None):
        raise NotImplementedError

    @classmethod
    def upload_document(cls, parsed_file):
        pass


class DummyDocument(Document):
    @classmethod
    def get_name(cls):
        return "dummy"

    @classmethod
    def get_scheme(cls):
        return {
            "columns": {
                "integer": {
                    "type": "integer"
                },
                "string": {
                    "type": "string"
                },
                "unix_time": {
                    "type": "integer"
                },
                "datetime": {
                    "type": "string"
                },
                "float": {
                    "type": "float"
                },
                # json is a raw json object.
                "json": {
                    "type": "json"
                }
            }
        }

    @classmethod
    def parse_document(cls, input_file, output_file, meta=None):
        with open(output_file, "w") as fd:
            dump({
                "rows": [
                    {
                        # "column name": "column value".
                        "integer": 12345,
                        "string": "12345",
                        "unix_time": 1636028330,  # unix timestamp.
                        "datetime": "2021-11-04T15:20:00",  # UTC timestamp.
                        #            YYYY-MM-DDTHH:MM:SS
                        "float": 12345.6789,
                    },
                    {
                        "integer": 12346,
                        "string": "12345-1",
                        "unix_time": 1636028330,
                        "datetime": "2021-11-04T15:20:00",
                    },
                    {
                        "integer": 12347,
                        "string": "12345-2",
                        "unix_time": 1636028330,
                        "datetime": "2021-11-04T15:20:00",
                        "json": {
                            "subinfo1": "subvalue1",
                            "subinfo2": 12345,
                        }
                    },
                ],
                "meta": meta or {},
            }, fd, indent=4)
        return None
