import openpyxl
from re import sub
from json import dump, load, loads
from os import environ
from os.path import basename
import yt.wrapper as yt
import logging
# import time

from .base import Document


class StartDocument(Document):
    @classmethod
    def get_name(cls):
        return 'start'

    @classmethod
    def get_yt_base(cls):
        return environ.get("GARAGE_PATH", "//home/carsharing/testing/data/garage")

    @classmethod
    def get_yt_directory(cls):
        yc = yt.YtClient(proxy="hahn", token=environ.get("YT_TOKEN"))
        if not yc.exists(cls.get_yt_base() + "/" + cls.get_name()):
            yc.mkdir(cls.get_yt_base() + "/" + cls.get_name())

        return cls.get_yt_base() + "/" + cls.get_name()

    @classmethod
    def get_scheme(cls):
        return {
            "columns": {
                "update_date": {
                    "type": "string",
                    "python_type": "datetime"
                },
                "model": {
                    "type": "string",
                    "python_type": "string"
                },
                "von": {
                    "type": "string",
                    "python_type": "string"
                },
                "vin": {
                    "type": "string",
                    "python_type": "string"
                },
                "status": {
                    "type": "string",
                    "python_type": "string"
                },
                "approximate_build_date": {
                    "type": "string",
                    "python_type": "datetime"
                },
                "approximate_delivery_date": {
                    "type": "string",
                    "python_type": "datetime"
                },
                "actual_delivery_date": {
                    "type": "string",
                    "python_type": "datetime"
                },
                "special_price": {
                    "type": "integer",
                    "python_type": "integer"
                },
                "contract_data": {
                    "type": "json"
                },
                "act_of_delivery_and_acceptance_date": {
                    "type": "json"
                }
            }
        }

    @classmethod
    def parse_document(cls, input_file, output_file, meta=None):
        logging.basicConfig(  # filename='app.log', filemode='a',
            level=logging.INFO,
            format='%(asctime)s - %(levelname)s - %(message)s', datefmt='%y/%b/%d %H:%M:%S')
        columns = {}
        rows = []
        data = {}
        scheme = cls.get_scheme()

        try:
            xl_file = openpyxl.open(input_file, read_only=True, data_only=True)
        except Exception as e:
            logging.error(f"Can not open {input_file}")
            raise e

        sheets = xl_file.sheetnames

        for sheet_name in sheets:
            sheet = xl_file[sheet_name]
            if len(scheme['columns']) != sheet.max_column:
                logging.warning('Number of columns in SCHEME and SHEET is different!' + '\n' +
                                '\t' + f'FILE: {input_file}' + '\n' +
                                '\t' + f'SHEET name: {sheet_name}' + '\n' +
                                '\t' + f'SHEET column count: {sheet.max_column}' + '\n' +
                                '\t' + f'SCHEME column count: {len(scheme["columns"])}')
            column_count = min(len(scheme['columns']), sheet.max_column)
            for row_number, row in enumerate(sheet.iter_rows(0, sheet.max_row), start=0):
                row_data = {}
                for col in range(0, column_count):
                    if row_number == 0:
                        if row[col].value is None:
                            columns[col] = None
                            continue
                        key = sub(r"\s", '_', str.strip(str(row[col].value)).lower())
                        columns[col] = key
                        continue

                    if row_number == 1:
                        if row[col].value in ['string', 'uint64', 'double']:
                            continue

                    if columns[col] is None:
                        continue

                    if row[col].value is None:
                        continue

                    col_type = scheme['columns'][columns[col]].get('python_type')
                    try:
                        if col_type == 'datetime':
                            row_data[columns[col]] = row[col].value.strftime('%Y-%m-%d')
                        elif col_type == 'integer':
                            row_data[columns[col]] = int(row[col].value)
                        elif col_type == 'float':
                            row_data[columns[col]] = float(row[col].value)
                        elif col_type == 'string':
                            row_data[columns[col]] = sub(r'\s+', ' ', str.strip(str(row[col].value)))
                        else:
                            row_data[columns[col]] = loads(row[col].value)
                    except Exception as e:
                        logging.error('Data type conversion' + '\n' + '\t' + f'FILE: {input_file}' + '\n' +
                                      '\t' + f'ROW: {row_number}' + '\n' + '\t' + f'COLUMN: "{columns[col]}"' + '\n ' +
                                      '\t' + f'VALUE: {row[col].value}' + '\n' +
                                      '\t' + f'SCHEME(target) data type: {col_type}')
                        raise e

                if row_number != 0 and len(row_data) > 0:
                    rows.append(row_data)

        data['rows'] = rows
        data["meta"] = meta or {}

        # start_time = time.time()
        with open(output_file, "w") as ofd:
            dump(data, ofd, indent=4)
        # logging.info(f'DUMPING: {time.time() - start_time}')

    @classmethod
    def upload_document(cls, parsed_file):
        def make_yt_type(type_in):
            if type_in == 'integer':
                return 'int64'
            elif type_in == 'float':
                return 'double'
            elif type_in == 'json':
                return 'any'
            else:
                return type_in

        yc = yt.YtClient(proxy="hahn", token=environ.get("YT_TOKEN"))
        with open(parsed_file, "r") as fd:
            data = load(fd)

            one_place = False
            if one_place:
                path = cls.get_yt_directory() + "/" + cls.get_name() + '_table'
            else:
                path = cls.get_yt_directory() + "/" + data.get("meta", {}).get("file_name", basename(parsed_file))
            schm = [{"name": k, "type": make_yt_type(v['type'])} for k, v in cls.get_scheme()['columns'].items()]

            if not yc.exists(path):
                yc.write_table(yt.TablePath(path, schema=schm), data["rows"], )
            else:
                yc.write_table(yt.TablePath(path, append=True), data["rows"], )


class LeasingDocument(StartDocument):
    @classmethod
    def get_name(cls):
        return 'leasing'

    @classmethod
    def get_scheme(cls):
        return {
            "columns": {
                "amort": {
                    "type": "float",
                    "python_type": "float"
                },
                "amount": {
                    "type": "float",
                    "python_type": "float"
                },
                "app": {
                    "type": "string",
                    "python_type": "datetime"
                },
                "buy_back_cost": {
                    "type": "float",
                    "python_type": "float"
                },
                "discount_rate": {
                    "type": "float",
                    "python_type": "float"
                },
                "end_date": {
                    "type": "string",
                    "python_type": "datetime"
                },
                "first_payment": {
                    "type": "float",
                    "python_type": "float"
                },
                "intro_asset": {
                    "type": "float",
                    "python_type": "float"
                },
                "intro_liability": {
                    "type": "float",
                    "python_type": "float"
                },
                "intro_payment": {
                    "type": "float",
                    "python_type": "float"
                },
                "l_type": {
                    "type": "string",
                    "python_type": "string"
                },
                "lessor": {
                    "type": "string",
                    "python_type": "string"
                },
                "liq_cost": {
                    "type": "float",
                    "python_type": "float"
                },
                "mod_date": {
                    "type": "string",
                    "python_type": "datetime"
                },
                "model": {
                    "type": "string",
                    "python_type": "string"
                },
                "removed": {
                    "type": "integer",
                    "python_type": "integer"
                },
                "rent": {
                    "type": "integer",
                    "python_type": "integer"
                },
                "start_date": {
                    "type": "string",
                    "python_type": "datetime"
                },
                "vin": {
                    "type": "string",
                    "python_type": "string"
                }
            }
        }


class FineDocument(StartDocument):
    @classmethod
    def get_name(cls):
        return 'fine'

    @classmethod
    def get_scheme(cls):
        return {
            "columns": {
                "acc_one": {
                    "type": "string",
                    "python_type": "string"
                },
                "decree": {
                    "type": "string",
                    "python_type": "string"
                },
                "noname": {
                    "type": "string",
                    "python_type": "string"
                },
                "acc_two": {
                    "type": "string",
                    "python_type": "string"
                },
                "payment": {
                    "type": "string",
                    "python_type": "string"
                }
            }
        }


class YandexIndividualsDocument(StartDocument):
    @classmethod
    def get_name(cls):
        return 'yandex_indiv'

    @classmethod
    def get_scheme(cls):
        return {
            "columns": {
                "decree": {
                    "type": "string",
                    "python_type": "string"
                },
                "date_decree": {
                    "type": "string",
                    "python_type": "datetime"
                },
                "date_exam": {
                    "type": "string",
                    "python_type": "datetime"
                },
                "fine": {
                    "type": "float",
                    "python_type": "float"
                },
                "sts": {
                    "type": "string",
                    "python_type": "string"
                },
                "vin": {
                    "type": "string",
                    "python_type": "string"
                },
                "number": {
                    "type": "string",
                    "python_type": "string"
                },
                "brand": {
                    "type": "string",
                    "python_type": "string"
                },
                "model": {
                    "type": "string",
                    "python_type": "string"
                },
                "contract_number": {
                    "type": "string",
                    "python_type": "string"
                },
                "refuse": {
                    "type": "string",
                    "python_type": "string"
                }
            }
        }
