package main

import (
	"context"
	"net"
	"net/http"
	"os"
	"os/signal"
	"sync"
	"syscall"

	"github.com/labstack/echo/v4"
	"github.com/labstack/echo/v4/middleware"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gobase/config"
	"a.yandex-team.ru/drive/analytics/gobase/core"
	"a.yandex-team.ru/drive/analytics/services/adjust"
	"a.yandex-team.ru/drive/analytics/services/backendlog"
	"a.yandex-team.ru/drive/analytics/services/garage"
	"a.yandex-team.ru/drive/analytics/services/licensechecks"
	"a.yandex-team.ru/drive/analytics/services/realtime"
	"a.yandex-team.ru/drive/analytics/services/tagshistory"
	"a.yandex-team.ru/drive/library/go/echolog"
	"a.yandex-team.ru/drive/library/go/gocfg"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/zootopia/analytics/drive/services/dyngeo"
	"a.yandex-team.ru/zootopia/analytics/drive/services/registry"

	_ "github.com/ClickHouse/clickhouse-go"
	_ "github.com/jackc/pgx/v4/stdlib"
	_ "github.com/mattn/go-sqlite3"
)

type Config struct {
	config.Config
	// Server contains server configuration.
	Server *ServerConfig `json:"server"`
	// Garage contains config for garage.
	Garage *garage.Config
	// SocketFile represents path to socket file.
	SocketFile string `json:"socket_file,omitempty"`
	// BackendLog contains BackendLog configuration.
	BackendLog *backendlog.Config `json:"backend_log"`
	// RealtimeWatcher contains RealtimeWatcher configuration.
	RealtimeWatcher *realtime.Config `json:"realtime_watcher"`
	// LicenseChecks contains LicenseChecks configuration.
	LicenseChecks *licensechecks.Config `json:"license_checks"`
}

// ServerConfig contains server configuration.
type ServerConfig struct {
	Addr string `json:"addr"`
}

// newServer should contain all initialization for server.
func newServer(c *core.Core, cfg *Config) *echo.Echo {
	e := echo.New()
	// Setup echo server.
	e.HideBanner, e.HidePort = true, true
	if c.Solomon != nil {
		e.Use(echolog.Solomon(c.Solomon, "server"))
	}
	e.Use(
		echolog.Logger(c.Logger("server")),
		middleware.Recover(),
		middleware.Gzip(),
	)
	e.Pre(middleware.RemoveTrailingSlash())
	// Register ping and health.
	e.GET("/ping", func(ctx echo.Context) error {
		return ctx.String(http.StatusOK, "pong")
	})
	e.GET("/health", func(ctx echo.Context) error {
		for _, db := range c.DBs {
			if err := db.PingContext(ctx.Request().Context()); err != nil {
				ctx.Logger().Error("Error:", err)
				return ctx.String(http.StatusInternalServerError, "unhealthy")
			}
		}
		for _, tvm := range c.TVMs {
			if _, err := tvm.GetStatus(ctx.Request().Context()); err != nil {
				ctx.Logger().Error("Error:", err)
				return ctx.String(http.StatusInternalServerError, "unhealthy")
			}
		}
		return ctx.String(http.StatusOK, "healthy")
	})
	// Register "/analytics" group.
	analyticsGroup := e.Group("/analytics")
	// Register TagsHistory service API
	tagsHistoryView := tagshistory.NewView(c)
	tagsHistoryGroup := analyticsGroup.Group(
		"/tags-history",
		c.ExtractYandexUser, c.ExtractDriveUser, c.ExtractDriveRoles,
	)
	tagsHistoryView.Register(tagsHistoryGroup)
	// Register Registry API.
	registryView := registry.NewView(c)
	registryGroup := analyticsGroup.Group(
		"/registry",
		c.ExtractYandexUser, c.ExtractDriveUser, c.ExtractDriveRoles,
	)
	registryView.Register(registryGroup)
	// Register license checks API.
	if cfg.LicenseChecks != nil {
		group := analyticsGroup.Group("/license-checks")
		licensechecks.NewView(c, *cfg.LicenseChecks).Register(group)
	}
	// Register Dynamic Geo Queries API.
	dynGeoView := dyngeo.NewView(c)
	dynGeoGroup := analyticsGroup.Group(
		"/dyngeo",
		c.ExtractYandexUser, c.ExtractDriveUser, c.ExtractDriveRoles,
	)
	dynGeoView.Register(dynGeoGroup)
	// Register marketing API.
	if c.Config.Adjust != nil {
		group := analyticsGroup.Group("/adjust")
		adjust.NewView(c).Register(group)
	}
	// Register garage API.
	if cfg.Garage != nil {
		group := analyticsGroup.Group(
			"/garage",
			c.ExtractYandexUser, c.ExtractDriveUser, c.ExtractDriveActions,
		)
		garage.NewView(c, *cfg.Garage).Register(group)
	}
	return e
}

// newSocketServer should contain all initialization for server.
func newSocketServer(c *core.Core, cfg *Config) *echo.Echo {
	e := echo.New()
	// Setup echo server.
	e.HideBanner, e.HidePort = true, true
	if c.Solomon != nil {
		e.Use(echolog.Solomon(c.Solomon, "socket"))
	}
	e.Use(
		echolog.Logger(c.Logger("socket")),
		middleware.Recover(),
		middleware.Gzip(),
	)
	e.Pre(middleware.RemoveTrailingSlash())
	g := e.Group("/socket")
	// Register ping and health.
	g.GET("/ping", func(ctx echo.Context) error {
		return ctx.String(http.StatusOK, "pong")
	})
	g.GET("/health", func(ctx echo.Context) error {
		for _, db := range c.DBs {
			if err := db.PingContext(ctx.Request().Context()); err != nil {
				ctx.Logger().Error("Error:", err)
				return ctx.String(http.StatusInternalServerError, "unhealthy")
			}
		}
		for _, tvm := range c.TVMs {
			if _, err := tvm.GetStatus(ctx.Request().Context()); err != nil {
				ctx.Logger().Error("Error:", err)
				return ctx.String(http.StatusInternalServerError, "unhealthy")
			}
		}
		return ctx.String(http.StatusOK, "healthy")
	})
	return e
}

var shutdown = make(chan os.Signal, 1)

func isError(err error) bool {
	return err != nil && err != http.ErrServerClosed
}

func serverMain(cmd *cobra.Command, _ []string) {
	configPath, err := cmd.Flags().GetString("config")
	if err != nil {
		panic(err)
	}
	cfg := Config{
		Config:     config.Config{LogLevel: log.InfoLevel},
		SocketFile: "/tmp/analytics-server.sock",
	}
	if err := gocfg.ParseFile(configPath, &cfg); err != nil {
		panic(err)
	}
	c, err := core.NewCore(&cfg.Config)
	if err != nil {
		panic(err)
	}
	logger := c.Logger("")
	if err := c.Start(); err != nil {
		logger.Fatal("Unable to start core", log.Error(err))
	}
	defer c.Stop()
	var waiter sync.WaitGroup
	defer waiter.Wait()
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	signal.Notify(shutdown, os.Interrupt, syscall.SIGTERM)
	waiter.Add(1)
	go func() {
		defer waiter.Done()
		select {
		case <-ctx.Done():
		case <-shutdown:
			cancel()
		}
	}()
	if file := cfg.SocketFile; file != "" {
		if err := os.Remove(file); err != nil && !os.IsNotExist(err) {
			panic(err)
		}
		srv := newSocketServer(c, &cfg)
		if srv.Listener, err = net.Listen("unix", file); err != nil {
			panic(err)
		}
		waiter.Add(1)
		go func() {
			defer waiter.Done()
			defer cancel()
			if err := srv.Start(""); isError(err) {
				logger.Error("Unable to start server", log.Error(err))
			}
		}()
		defer func() {
			if err := srv.Shutdown(context.Background()); err != nil {
				logger.Error("Unable to stop server", log.Error(err))
			}
		}()
	}
	if cfg.Server != nil {
		srv := newServer(c, &cfg)
		waiter.Add(1)
		go func() {
			defer waiter.Done()
			defer cancel()
			if err := srv.Start(cfg.Server.Addr); isError(err) {
				logger.Error("Unable to start server", log.Error(err))
			}
		}()
		defer func() {
			if err := srv.Shutdown(context.Background()); err != nil {
				logger.Error("Unable to stop server", log.Error(err))
			}
		}()
	}
	if cfg.BackendLog != nil {
		backendlog.NewService(c, *cfg.BackendLog).Start()
	}
	if cfg.RealtimeWatcher != nil {
		realtime.NewWatcher(c, *cfg.RealtimeWatcher).Start()
	}
	<-ctx.Done()
}

func main() {
	cmd := cobra.Command{}
	cmd.PersistentFlags().String("config", "config.json", "")
	serverCmd := cobra.Command{Use: "server", Run: serverMain}
	cmd.AddCommand(&serverCmd)
	if err := cmd.Execute(); err != nil {
		panic(err)
	}
}
