# -*- coding: utf-8 -*-

from logging import getLogger
from sqlalchemy.orm.attributes import flag_modified
from drive.analytics.pybase.config import YT
from drive.analytics.pybase.models import session
from drive.library.py.thiefs import (
    AndroidDevice, BelkacarClient, BelkacarToken, TokenExpiredError,
)
from drive.library.py.time import now
from .utils import solomon, get_animals_session, new_record, get_random_account


_log = getLogger(__name__)


def _load_city_cars(args, city):
    labels = {"provider": "belkacar", "action": "cars", "city": city}
    _log.info("Parsing Belkacar for city '{}'...".format(city))
    solomon.signal("begin", 1, labels)
    begin_time = now()
    http_client = get_animals_session(args)
    client = BelkacarClient(client=http_client)
    account = get_random_account("belkacar", city)
    device = AndroidDevice.unmarshal_dict(account.data["device"])
    token = BelkacarToken.unmarshal_dict(account.data["token"])
    phone_number = account.data["phone_number"]
    location = (
        account.data["home_location"]["longitude"],
        account.data["home_location"]["latitude"],
    )
    try:
        cars = client.get_cars(device, location, token)
    except TokenExpiredError:
        _log.info("Token expired for account '{}'.".format(account.id))
        token = client.refresh_token(device, location, phone_number, token)
        account.data["token"] = token.marshal_dict()
        session.add(account)
        flag_modified(account, "data")
        session.commit()
        cars = client.get_cars(device, location, token)
    cars_by_type = {}
    for car in cars["list"]:
        cars_by_type[car["type_id"]] = car
    offers = {}
    for car_type, car in cars_by_type.items():
        offer = client.get_car_info(device, location, token, car["id"])
        offers[car_type] = offer
    for car in cars["list"]:
        car["offer"] = offers.get(car["type_id"], None)
    record = new_record(
        table="{}/data/thief/belkacar/raw_data".format(YT["home_path"]),
        name="cars/v2",
        result=cars,
        params={"city": city},
    )
    session.add(record)
    session.commit()
    solomon.signal("total_cars", len(cars["list"]), labels)
    duration = now() - begin_time
    _log.info("Parser finished.")
    solomon.signal("end", 1, labels)
    solomon.signal("duration", duration.total_seconds(), labels)


def main(args):
    for city in ("moscow", "sochi"):
        try:
            _load_city_cars(args, city)
        except Exception as exc:
            _log.exception(exc)
    return 0
