# -*- coding: utf-8 -*-

from logging import getLogger
from multiprocessing.dummy import Pool as ThreadPool
from drive.analytics.pybase.config import YT
from drive.analytics.pybase.models import session
from drive.library.py.thiefs import TransportClient
from drive.library.py.time import now, to_timestamp
from .utils import solomon, get_animals_session, new_record


_log = getLogger(__name__)


def main(args):
    labels = {
        "provider": "transport",
        "action": "cars_info" if args.load_cars_info else "cars",
        "city": "moscow",
    }
    _log.info("Parsing Moscow Transport...")
    solomon.signal("begin", 1, labels)
    begin_time = now()
    http_client = get_animals_session(args)
    client = TransportClient(client=http_client)

    def load_tile(tile_bounds):
        resp = client.get_cars(tile_bounds)
        cars = resp.get("cars", [])
        result = list()
        if not args.load_cars_info:
            result.append(dict(
                cars=cars,
                ts=int(to_timestamp(now())),
            ))
        else:
            for car in cars:
                try:
                    resp = client.get_car_info(car["id"])
                    result.append(resp)
                except Exception as exc:
                    _log.exception(exc)
        return result

    moscow_bounds = (
        (37.0, 55.0),  # left bottom (lon, lat)
        (38.5, 56.0),  # right upper (lon, lat)
    )
    d_lon, d_lat = 0.015, 0.01
    pos = list(moscow_bounds[0])
    result = list()
    total_cars = 0
    try:
        futures = list()
        pool = ThreadPool(16)
        while pos[0] < moscow_bounds[1][0]:
            pos[1] = moscow_bounds[0][1]
            while pos[1] < moscow_bounds[1][1]:
                tile_bounds = (
                    (pos[0], pos[1]),
                    (pos[0] + d_lon, pos[1] + d_lat),
                )
                futures.append(pool.apply_async(load_tile, [tile_bounds]))
                pos[1] += d_lat
            pos[0] += d_lon
        for future in futures:
            try:
                resps = future.get(timeout=10)
                for resp in resps:
                    result.append(resp)
                    total_cars += 1 if args.load_cars_info else len(resp)
            except Exception as exc:
                _log.exception(exc)
    finally:
        pool.close()
        pool.join()
    _log.info("Found {} cars".format(total_cars))
    record = new_record(
        table="{}/data/thief/transport/raw_data".format(YT["home_path"]),
        name="cars_info/v2" if args.load_cars_info else "cars/v2",
        result=result,
    )
    session.add(record)
    session.commit()
    solomon.signal("total_cars", total_cars, labels)
    duration = now() - begin_time
    _log.info("Parser finished.")
    solomon.signal("end", 1, labels)
    solomon.signal("duration", duration.total_seconds(), labels)
    return 0
