package models

import (
	"context"
	"database/sql"
	"database/sql/driver"
	"fmt"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/yt/go/yson"
	"a.yandex-team.ru/zootopia/library/go/db"
	"a.yandex-team.ru/zootopia/library/go/db/events"
	"a.yandex-team.ru/zootopia/library/go/db/objects"
)

type baseStore struct {
	db      *sql.DB
	objects objects.Store
	events  events.Store
}

// newBaseStore creates a new base store.
func newBaseStore(
	conn *sql.DB, dialect db.Dialect,
	object objects.Object, objectID, objectTable string,
	event events.Event, eventID, eventTable string,
) *baseStore {
	return &baseStore{
		db:      conn,
		objects: objects.NewStore(object, objectID, objectTable, dialect),
		events:  events.NewStore(event, eventID, eventTable, dialect),
	}
}

type baseCachedStore struct {
	store    *baseStore
	cache    *sql.DB
	consumer events.Consumer
	initFunc func(tx *sql.Tx) error
}

func (s *baseCachedStore) Init(ctx context.Context) error {
	return gosql.WithTxContext(
		ctx, s.store.db, &sql.TxOptions{ReadOnly: true},
		func(tx *sql.Tx) error {
			if err := s.initEvents(ctx, tx); err != nil {
				return err
			}
			return s.initObjects(ctx, tx)
		},
	)
}

func (s *baseCachedStore) Sync(ctx context.Context) error {
	return fmt.Errorf("not implemented")
}

const oldGapDistance = 2000

func (s *baseCachedStore) initEvents(_ context.Context, tx *sql.Tx) error {
	lastID, err := s.store.events.LastEventID(tx)
	if err != nil {
		// LastEventID can return sql.ErrNoRows if there is no events.
		if err != sql.ErrNoRows {
			return err
		}
		lastID = 1
	}
	// lastID should be always greater than 0.
	if lastID > oldGapDistance {
		lastID -= oldGapDistance
	} else {
		lastID = 1
	}
	// Create consumer with delayed begin EventID for graceful
	// detection of all gaps.
	s.consumer = events.NewConsumer(s.store.events, lastID)
	return s.consumer.ConsumeEvents(tx, ignoreEvent)
}

func (s *baseCachedStore) initObjects(ctx context.Context, tx *sql.Tx) error {
	if err := gosql.WithTxContext(
		ctx, s.cache, nil, s.initFunc,
	); err != nil {
		return err
	}
	rows, err := s.store.objects.ReadObjects(tx)
	if err != nil {
		return err
	}
	defer func() {
		_ = rows.Close()
	}()
	if err := gosql.WithTxContext(
		ctx, s.cache, nil,
		func(tx *sql.Tx) error {
			for rows.Next() {
				if _, err := s.store.objects.CreateObject(
					tx, rows.Object(),
				); err != nil {
					return err
				}
			}
			return nil
		},
	); err != nil {
		return err
	}
	return rows.Err()
}

// newBaseCachedStore creates a new cached store.
//
// DB connection for cache should only use SQLite driver. Otherwise
// there will be problems with queries.
func newBaseCachedStore(
	store *baseStore, cache *sql.DB, initFunc func(tx *sql.Tx) error,
) *baseCachedStore {
	return &baseCachedStore{
		store:    store,
		cache:    cache,
		initFunc: initFunc,
	}
}

// ignoreEvent does nothing.
func ignoreEvent(events.Event) error {
	return nil
}

// baseEvent represents basic event for objects.
type baseEvent struct {
	HistoryEventID      int64   `db:"history_event_id" yson:"history_event_id"`
	HistoryAction       string  `db:"history_action" yson:"history_action"`
	HistoryTimestamp    int64   `db:"history_timestamp" yson:"history_timestamp"`
	HistoryUserID       string  `db:"history_user_id" yson:"history_user_id"`
	HistoryOriginatorID NString `db:"history_originator_id" yson:"history_originator_id"`
	HistoryComment      NString `db:"history_comment" yson:"history_comment"`
}

// EventID returns ID of event.
func (e baseEvent) EventID() int64 {
	return e.HistoryEventID
}

// EventAction returns actions of event.
func (e baseEvent) EventAction() string {
	return e.HistoryAction
}

// EventTime returns time of event.
func (e baseEvent) EventTime() time.Time {
	return time.Unix(e.HistoryTimestamp, 0)
}

type NInt64 int64

func (v NInt64) Value() (driver.Value, error) {
	if v == 0 {
		return nil, nil
	}
	return int64(v), nil
}

func (v *NInt64) Scan(value interface{}) error {
	switch x := value.(type) {
	case nil:
		*v = 0
	case int64:
		*v = NInt64(x)
	default:
		return fmt.Errorf("unsupported value: %T", x)
	}
	return nil
}

func (v *NInt64) UnmarshalYSON(data []byte) error {
	var value *int64
	if err := yson.Unmarshal(data, &value); err != nil {
		return err
	}
	if value == nil {
		*v = 0
	} else {
		*v = NInt64(*value)
	}
	return nil
}

type NUInt64 uint64

func (v NUInt64) Value() (driver.Value, error) {
	if v == 0 {
		return nil, nil
	}
	return uint64(v), nil
}

func (v *NUInt64) Scan(value interface{}) error {
	switch x := value.(type) {
	case nil:
		*v = 0
	case int64:
		*v = NUInt64(x)
	case uint64:
		*v = NUInt64(x)
	default:
		return fmt.Errorf("unsupported value: %T", x)
	}
	return nil
}

func (v *NUInt64) UnmarshalYSON(data []byte) error {
	var value *uint64
	if err := yson.Unmarshal(data, &value); err != nil {
		return err
	}
	if value == nil {
		*v = 0
	} else {
		*v = NUInt64(*value)
	}
	return nil
}

type NString string

func (v NString) Value() (driver.Value, error) {
	if v == "" {
		return nil, nil
	}
	return string(v), nil
}

func (v *NString) Scan(value interface{}) error {
	switch x := value.(type) {
	case nil:
		*v = ""
	case []uint8:
		*v = NString(x)
	case string:
		*v = NString(x)
	default:
		return fmt.Errorf("unsupported value: %T", x)
	}
	return nil
}

func (v *NString) UnmarshalYSON(data []byte) error {
	var value *string
	if err := yson.Unmarshal(data, &value); err != nil {
		return err
	}
	if value == nil {
		*v = ""
	} else {
		*v = NString(*value)
	}
	return nil
}

type NTime struct {
	time.Time
}

func (v NTime) Value() (driver.Value, error) {
	if v.IsZero() {
		return nil, nil
	}
	return driver.Value(v.Time), nil
}

func (v *NTime) Scan(value interface{}) error {
	switch x := value.(type) {
	case nil:
		*v = NTime{}
	case time.Time:
		v.Time = x
	default:
		return fmt.Errorf("unsupported value: %T", x)
	}
	return nil
}

type NUUID struct {
	uuid.UUID
}

func (v NUUID) Value() (driver.Value, error) {
	if v.UUID == uuid.Nil {
		return nil, nil
	}
	return v.UUID.Value()
}

func (v *NUUID) Scan(value interface{}) error {
	switch value.(type) {
	case nil:
		v.UUID = uuid.Nil
	default:
		return v.UUID.Scan(value)
	}
	return nil
}

type NBool bool

func (v NBool) Value() (driver.Value, error) {
	return bool(v), nil
}

func (v *NBool) Scan(value interface{}) error {
	switch x := value.(type) {
	case nil:
		*v = false
	case bool:
		*v = NBool(x)
	default:
		return fmt.Errorf("unsupported value: %T", x)
	}
	return nil
}

func (v NBool) MarshalYSON() ([]byte, error) {
	return yson.Marshal(bool(v))
}

func (v *NBool) UnmarshalYSON(data []byte) error {
	var value *bool
	if err := yson.Unmarshal(data, &value); err != nil {
		return err
	}
	if value == nil {
		*v = false
	} else {
		*v = NBool(*value)
	}
	return nil
}
