package models

import (
	"bytes"
	"compress/zlib"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"io/ioutil"

	"github.com/golang/protobuf/proto"

	fb "a.yandex-team.ru/drive/backend/proto"
	pl "a.yandex-team.ru/drive/telematics/server/location/proto"
)

// CompiledRide represents compiled ride.
type CompiledRide struct {
	baseEvent
	SessionID string  `db:"session_id" yson:"session_id"`
	ObjectID  NUUID   `db:"object_id" yson:"object_id"`
	Price     int64   `db:"price" yson:"price"`
	Duration  int     `db:"duration" yson:"duration"`
	Start     int64   `db:"start" yson:"start"`
	Finish    int64   `db:"finish" yson:"finish"`
	Meta      NString `db:"meta" yson:"meta"`
	MetaProto NString `db:"meta_proto" yson:"meta_proto"`
	HardProto NString `db:"hard_proto" yson:"hard_proto"`
}

type compiledRideMetaLocation struct {
	Latitude  float64 `json:"latitude"`
	Longitude float64 `json:"longitude"`
}

type compiledRideMetaBill struct {
	Records []struct {
		ID       string `json:"id"`
		Title    string `json:"title"`
		Type     string `json:"type"`
		Cost     int32  `json:"cost"`
		Duration uint32 `json:"duration"`
		Details  string `json:"details"`
	} `json:"records"`
	FreeDuration   *uint32 `json:"free_duration"`
	PricedDuration *uint32 `json:"priced_duration"`
}

type compiledRideMeta struct {
	Diff *struct {
		Start   *compiledRideMetaLocation `json:"start"`
		Finish  *compiledRideMetaLocation `json:"finish"`
		Mileage float64                   `json:"mileage"`
	} `json:"diff"`
	RawBill json.RawMessage       `json:"bill"`
	Bill    *compiledRideMetaBill `json:"-"`
}

type CompiledRideData struct {
	*fb.TCompiledRidingHard
	*fb.TSnapshotsDiff
}

// ParseData parses compiled ride data.
func (r CompiledRide) ParseData() (CompiledRideData, error) {
	if r.HardProto == "" {
		metaProto, err := r.parseDataV2()
		if err != nil {
			return CompiledRideData{}, err
		}
		var hardProto fb.TCompiledRidingHard
		if metaProto.Bill != nil {
			hardProto.Bill = metaProto.Bill
		}
		if metaProto.Offer != nil {
			hardProto.Offer = metaProto.Offer
		}
		if metaProto.LocalEvents != nil {
			hardProto.LocalEvents = &fb.TCompiledLocalEvents{
				Events: metaProto.LocalEvents,
			}
		}
		return CompiledRideData{
			TCompiledRidingHard: &hardProto,
			TSnapshotsDiff:      metaProto.SnapshotsDiff,
		}, nil
	}
	// Parse v3 format.
	var metaProto fb.TCompiledRidingMeta
	if err := parseProtoBase64(string(r.MetaProto), &metaProto); err != nil {
		return CompiledRideData{}, err
	}
	var hardProto fb.TCompiledRidingHard
	if err := parseProtoBase64(string(r.HardProto), &hardProto); err != nil {
		return CompiledRideData{}, err
	}
	return CompiledRideData{
		TCompiledRidingHard: &hardProto,
		TSnapshotsDiff:      metaProto.SnapshotsDiff,
	}, nil
}

func (r CompiledRide) parseDataV2() (*fb.TCompiledRidingMeta, error) {
	if r.MetaProto == "" {
		meta, err := r.parseDataV1()
		if err != nil {
			return &fb.TCompiledRidingMeta{}, err
		}
		var metaProto fb.TCompiledRidingMeta
		if meta.Diff != nil {
			var diff fb.TSnapshotsDiff
			if meta.Diff.Start != nil {
				diff.Start = &pl.TLocation{}
			}
			if meta.Diff.Finish != nil {
				diff.Last = &pl.TLocation{}
			}
			mileage := float32(meta.Diff.Mileage)
			diff.Mileage = &mileage
			trueValue := true
			diff.Finished = &trueValue
			metaProto.SnapshotsDiff = &diff
		}
		if meta.Bill != nil {
			var bill fb.TBill
			bill.FreeDuration = meta.Bill.FreeDuration
			bill.PricedDuration = meta.Bill.PricedDuration
			for _, record := range meta.Bill.Records {
				protoRecord := fb.TBillRecord{}
				protoRecord.Type = &record.Type
				switch record.Type {
				case "old_state_reservation", "old_state_parking",
					"old_state_riding", "old_state_acceptance":
					cost := record.Cost
					if cost < 0 {
						cost = -cost
					}
					protoRecord.Cost = &cost
					protoRecord.Duration = &record.Duration
				case "discount":
					protoRecord.Id = &record.ID
					protoRecord.Title = &record.Title
					fallthrough
				case "billing_bonus":
					cost := record.Cost
					if cost > 0 {
						cost = -cost
					}
					protoRecord.Cost = &cost
					protoRecord.Details = &record.Details
				case "overrun", "overrunning":
					overrunValue := "overrun"
					protoRecord.Type = &overrunValue
					fallthrough
				case "overtime", "pack":
					cost := record.Cost
					if cost < 0 {
						cost = -cost
					}
					protoRecord.Cost = &cost
				case "total":
				default:
					return &metaProto, fmt.Errorf(
						"unsupported type: %q", record.Type,
					)
				}
				bill.Record = append(bill.Record, &protoRecord)
			}
			metaProto.Bill = &bill
		}
		return &metaProto, nil
	}
	// Parse v2 format.
	var metaProto fb.TCompiledRidingMeta
	err := parseProtoBase64(string(r.MetaProto), &metaProto)
	return &metaProto, err
}

func (r CompiledRide) parseDataV1() (compiledRideMeta, error) {
	// Parse v1 format.
	var meta compiledRideMeta
	if err := json.Unmarshal([]byte(r.Meta), &meta); err != nil {
		return compiledRideMeta{}, err
	}
	if err := json.Unmarshal(meta.RawBill, &meta.Bill); err != nil {
		meta.Bill = &compiledRideMetaBill{}
		if err := json.Unmarshal(
			meta.RawBill, &meta.Bill.Records,
		); err != nil {
			return compiledRideMeta{}, err
		}
	}
	return meta, nil
}

func parseProtoBase64(data string, message proto.Message) error {
	bytes, err := base64.StdEncoding.DecodeString(data)
	if err != nil {
		return err
	}
	// zlib fix.
	bytes = zlibCompressFix(bytes)
	return proto.Unmarshal(bytes, message)
}

func zlibCompressFix(data []byte) []byte {
	r, err := zlib.NewReader(bytes.NewReader(data))
	if err != nil {
		return data
	}
	defer r.Close()
	newData, err := ioutil.ReadAll(r)
	if err != nil {
		return data
	}
	return newData
}
