package models

import (
	"fmt"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/drive/analytics/goback/models/tags"
	pdb "a.yandex-team.ru/drive/backend/database/drive/proto"
	pp "a.yandex-team.ru/drive/backend/proto"
)

type Tag struct {
	ID        uuid.UUID `db:"tag_id" yson:"tag_id"`
	ObjectID  string    `db:"object_id" yson:"object_id"`
	Tag       NString   `db:"tag" yson:"tag"`
	Data      NString   `db:"data" yson:"data"`
	Performer NString   `db:"performer" yson:"performer"`
	Priority  NInt64    `db:"priority" yson:"priority"`
	Snapshot  NString   `db:"snapshot" yson:"snapshot"`
}

type TagEvent struct {
	baseEvent
	Tag
}

func (o Tag) ScanData(data tags.TagData) error {
	var blob []byte
	var head pdb.TTagHeader
	if err := tags.ParseBlobWithHeader(string(o.Data), &blob, &head); err != nil {
		return err
	}
	if len(blob) == 0 {
		return nil
	}
	if head.GetIsTagProto() {
		if err := data.Unmarshal(blob); err != nil {
			return fmt.Errorf("error scanning tag %T: %s", data, err)
		}
	} else {
		if err := data.UnmarshalJSON(blob); err != nil {
			return fmt.Errorf("error scanning tag %T: %s", data, err)
		}
	}
	return nil
}

func (o Tag) ParseData(typ string) (tags.TagData, error) {
	data, ok := tags.NewTagData(typ)
	if !ok {
		return nil, fmt.Errorf("tag of type %q does not exitsts", typ)
	}
	if err := o.ScanData(data); err != nil {
		return nil, err
	}
	return data, nil
}

func (o Tag) ParseSnapshot() (tags.Snapshot, error) {
	var blob []byte
	var head pp.TSnapshotHeader
	if err := tags.ParseBlobWithHeader(string(o.Snapshot), &blob, &head); err != nil {
		return nil, err
	}
	if head.Type == nil {
		return nil, fmt.Errorf("empty snapshot type")
	}
	snapshot, ok := tags.NewTagSnapshot(*head.Type)
	if !ok {
		return nil, fmt.Errorf(
			"tag of type %q does not exitsts", *head.Type,
		)
	}
	if err := snapshot.Unmarshal(blob); err != nil {
		return nil, err
	}
	return snapshot, nil
}
