package tags

import (
	"bytes"
	"encoding/base64"
	"encoding/binary"
	"encoding/json"
	"fmt"
	"reflect"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/drive/backend/processors/car_scanner/proto"

	pd "a.yandex-team.ru/drive/backend/data/proto"
	pdb "a.yandex-team.ru/drive/backend/database/drive/proto"
	pf "a.yandex-team.ru/drive/backend/proto"
)

type TagData interface {
	proto.Unmarshaler
	json.Unmarshaler
}

type Snapshot interface {
	proto.Unmarshaler
}

// NewTagData creates a new instance of TagData.
//
// Returns false, if data type with specified type does not exists.
func NewTagData(typ string) (TagData, bool) {
	data, ok := tagDataTypes[typ]
	if !ok || data == nil {
		return nil, false
	}
	// Extract type of data.
	dataType := reflect.TypeOf(data).Elem()
	// Create new instance of data.
	return reflect.New(dataType).Interface().(TagData), true
}

// NewTagSnapshot creates a new instance of Snapshot.
//
// Returns false, if snapshot type with specified type does not exists.
func NewTagSnapshot(typ string) (Snapshot, bool) {
	data, ok := snapshotTypes[typ]
	if !ok || data == nil {
		return nil, false
	}
	// Extract type of snapshot.
	dataType := reflect.TypeOf(data).Elem()
	// Create new instance of snapshot.
	return reflect.New(dataType).Interface().(Snapshot), true
}

type BlobWithTagHeader struct {
	Header *pdb.TTagHeader
	Blob   []byte
}

type BlobWithSnapshotHeader struct {
	Header *pf.TSnapshotHeader
	Blob   []byte
}

func ParseBlobWithTagHeader(data string) (bwh BlobWithTagHeader, err error) {
	raw, err := base64.StdEncoding.DecodeString(data)
	if err != nil {
		return
	}
	reader := bytes.NewReader(raw)
	var size int32
	if err = binary.Read(reader, binary.LittleEndian, &size); err != nil {
		return
	}
	header := make([]byte, size)
	if _, err = reader.Read(header); err != nil {
		return
	}
	bwh.Header = &pdb.TTagHeader{}
	if err = proto.Unmarshal(header, bwh.Header); err != nil {
		return
	}
	bwh.Blob = make([]byte, reader.Len())
	if _, err = reader.Read(bwh.Blob); err != nil {
		return
	}
	return
}

func ParseBlobWithSnapshotHeader(
	data string,
) (bwh BlobWithSnapshotHeader, err error) {
	raw, err := base64.StdEncoding.DecodeString(data)
	if err != nil {
		return
	}
	reader := bytes.NewReader(raw)
	var size int32
	if err = binary.Read(reader, binary.LittleEndian, &size); err != nil {
		return
	}
	header := make([]byte, size)
	if _, err = reader.Read(header); err != nil {
		return
	}

	bwh.Header = &pf.TSnapshotHeader{}
	if err = proto.Unmarshal(header, bwh.Header); err != nil {
		return
	}
	bwh.Blob = make([]byte, reader.Len())
	if _, err = reader.Read(bwh.Blob); err != nil {
		return
	}
	return
}

func ParseBlobWithHeader(
	data string, blob *[]byte, head proto.Message,
) error {
	raw, err := base64.StdEncoding.DecodeString(data)
	if err != nil {
		return err
	}
	reader := bytes.NewReader(raw)
	var size int32
	if err := binary.Read(reader, binary.LittleEndian, &size); err != nil {
		return err
	}
	header := make([]byte, size)
	if _, err := reader.Read(header); err != nil {
		return err
	}
	if err := proto.Unmarshal(header, head); err != nil {
		return err
	}
	*blob = make([]byte, reader.Len())
	if _, err := reader.Read(*blob); err != nil {
		return err
	}
	return nil
}

var tagDataTypes = map[string]TagData{
	"chargable_tag":                 &ChargableTagData{},
	"temp_action":                   &TempActionTagData{},
	"simple_user_tag":               &SimpleUserTagData{},
	"unique_user_tag":               &UniqueUserTagData{},
	"rewrite_user_tag":              &RewriteUserTagData{},
	"user_problem_tag":              &UserProblemTagData{},
	"user_registration_tag":         &UserRegistrationTagData{},
	"user_push":                     &UserPushTagData{},
	"user_landing":                  &UserLandingTagData{},
	"operation_tag":                 &OperationTagData{},
	"refund_tag":                    &RefundTagData{},
	"fixed_sum_tag":                 &FixedSumTagData{},
	"cluster_tag":                   &ClusterTagData{},
	"additional_feature_tag":        &AdditionalFeatureTagData{},
	"car_scanner":                   &CarScannerTagData{},
	"area_stand_registration_promo": &AreaStandRegistrationPromoTagData{},
	"simple_fueling_tag":            &SimpleFuelingTagData{},
	"telematics_deferred_command":   &TelematicsDeferredCommandTagData{},
	"user_mail_notification_tag":    nil,
	"fueling_order":                 &FuelingOrderTagData{},
	"user_robot_call_tag":           &UserRobotCallTagData{},
	"fix_point_area_correction":     &FixPointAreaCorrectionTagData{},
	"user_connection_tag":           &UserConnectionTagData{},
	"transformation":                &TransformationTagData{},
	"dictionary_tag":                &DictionaryTagData{},
	"user_dictionary_tag":           &UserDictionaryTagData{},
	"car_maintenance_tag":           nil,
	"service_fueling_tag":           &ServiceFuelingTagData{},
	"coloring_tag":                  nil,
	"string_field_tag":              nil,
	"unique_tag_rewrite":            nil,
	"telematics_firmware":           &TelematicsFirmwareTagData{},
	"user_support_mail_tag":         &UserSupportMailTagData{},
	"user_fueling_tag":              &UserFuelingTagData{},
	"confirmable_tag":               &ConfirmableTagData{},
	"dry_run_parking_payment":       nil,
	"billing_tag":                   nil,
	"car_repair_tag":                &RepairTagData{},
	"no_signal":                     nil,
	"feedback_trace_tag":            nil,
	"car_service_tag":               &CarServiceTagData{},
	"incorrect_v":                   nil,
	"operation_area_tag":            nil,
	"fueling_result":                nil,
	"unique_tag":                    nil,
	"properties_patch_tag":          nil,
	"low_voltage":                   nil,
	"tag_reservation_features":      nil,
	"bonus_tag":                     nil,
	"area_potential":                &AreaPotentialTagData{},
	"simple_car_tag":                &SimpleCarTagData{},
	"parking_payment":               nil,
	"tag_reservation_futures":       nil,
	"area_user_info":                nil,
	"user_alert_tag":                nil,
	"external_promo_tag":            nil,
	"ticket_tag":                    nil,
	"incorrect_moving":              nil,
	"telematics":                    &TelematicsTagData{},
	"multi_user_tag":                nil,
	"staff_hierarchy":               nil,
	"model_tag":                     nil,
	"incorrect_rzone":               nil,
	"user_support_chat_tag":         nil,
	"user_support_call_tag":         nil,
	"telematics_configuration":      &TelematicsConfigurationTagData{},
	"feedback_device_tag":           nil,
	"documents_reask_tag":           nil,
	"fixed_bonus_tag":               nil,
	"transportation_tag":            nil,
	"car_dictionary_tag":            &CarDictionaryTagData{},
	"express_cleaning":              nil,
	"car_alert_tag":                 nil,
	"photo_request":                 &PhotoRequestTagData{},
	"user_origin":                   &UserOriginTagData{},
	"scoring_user_tag":              &ScoringTagData{},
	"scoring_trace_tag":             &ScoringTagData{},
}

var snapshotTypes = map[string]Snapshot{
	"device_snapshot": &DeviceSnapshot{},
}

var ErrFormatNotSupported = fmt.Errorf("format is not supported")

type ChargableTagData struct {
	pf.TChargableTag
}

type chargableTagDataJSON struct {
	Offer struct {
		Time        uint32 `json:"timestamp"`
		Deadline    uint32 `json:"deadline"`
		OfferID     string `json:"offer_id"`
		UserID      string `json:"user_id"`
		ObjectID    string `json:"object_id"`
		Name        string `json:"name"`
		FromScanner bool   `json:"from_scanner"`
		Type        string `json:"type"`
		Data        struct {
			PricingBySeconds bool `json:"pricing_by_seconds"`
			Prices           struct {
				Riding  uint32 `json:"riding"`
				Parking uint32 `json:"parking"`
			} `json:"prices"`
		} `json:"data"`
	} `json:"offer"`
}

func (d *ChargableTagData) UnmarshalJSON(data []byte) error {
	var jsonData chargableTagDataJSON
	if err := json.Unmarshal(data, &jsonData); err != nil {
		return err
	}
	d.Offer = &pf.TOffer{}
	d.Offer.Timestamp = &jsonData.Offer.Time
	d.Offer.Deadline = &jsonData.Offer.Deadline
	d.Offer.OfferId = &jsonData.Offer.OfferID
	d.Offer.UserId = &jsonData.Offer.UserID
	d.Offer.ObjectId = &jsonData.Offer.ObjectID
	d.Offer.Name = &jsonData.Offer.Name
	d.Offer.FromScanner = &jsonData.Offer.FromScanner
	if jsonData.Offer.Type == "" {
		jsonData.Offer.Type = "standart_offer"
	}
	d.Offer.InstanceType = &jsonData.Offer.Type
	d.Offer.StandartOffer = &pf.TStandartOffer{}
	d.Offer.StandartOffer.PricingBySeconds =
		&jsonData.Offer.Data.PricingBySeconds
	d.Offer.StandartOffer.PriceRiding = &jsonData.Offer.Data.Prices.Riding
	d.Offer.StandartOffer.PriceParking = &jsonData.Offer.Data.Prices.Parking
	return nil
}

func (d *ChargableTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TChargableTag)
}

type SimpleUserTagData struct {
	pf.TCommonTagData
}

func (d *SimpleUserTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *SimpleUserTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TCommonTagData)
}

type (
	UniqueUserTagData        = SimpleUserTagData
	RewriteUserTagData       = SimpleUserTagData
	AdditionalFeatureTagData = SimpleUserTagData
	ConfirmableTagData       = SimpleUserTagData
	SimpleFuelingTagData     = SimpleUserTagData
	ServiceFuelingTagData    = SimpleFuelingTagData
)

type UserProblemTagData struct {
	pd.TUserProblemTagData
}

func (d *UserProblemTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *UserProblemTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TUserProblemTagData)
}

type (
	OperationTagData = UserProblemTagData
	RefundTagData    = UserProblemTagData
	FixedSumTagData  = UserProblemTagData
)

type UserRegistrationTagData struct {
	pd.TRegistrationUserTagData
}

func (d *UserRegistrationTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *UserRegistrationTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TRegistrationUserTagData)
}

type UserPushTagData struct {
	pd.TUserMessageTagData
}

func (d *UserPushTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *UserPushTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TUserMessageTagData)
}

type UserLandingTagData struct {
	pd.TLandingUserTagData
}

func (d *UserLandingTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *UserLandingTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TLandingUserTagData)
}

type UserRobotCallTagData struct {
	pd.TUserRobotCallTagData
}

func (d *UserRobotCallTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *UserRobotCallTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TUserRobotCallTagData)
}

type UserConnectionTagData struct {
	pd.TConnectionUserTagData
}

func (d *UserConnectionTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *UserConnectionTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TConnectionUserTagData)
}

type UserFuelingTagData struct {
	pd.TUserFuelingTag
}

func (d *UserFuelingTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *UserFuelingTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TUserFuelingTag)
}

type UserSupportMailTagData struct {
	pd.TSupportEmailTagData
}

func (d *UserSupportMailTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *UserSupportMailTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TSupportEmailTagData)
}

type TempActionTagData struct {
	pd.TTemporaryActionTag
}

func (d *TempActionTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *TempActionTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TTemporaryActionTag)
}

type AreaStandRegistrationPromoTagData struct {
	pd.TStandRegistrationPromoTag
}

func (d *AreaStandRegistrationPromoTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *AreaStandRegistrationPromoTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TStandRegistrationPromoTag)
}

type FuelingOrderTagData struct {
	pf.TFuelingOrderTag
}

func (d *FuelingOrderTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *FuelingOrderTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TFuelingOrderTag)
}

type FixPointAreaCorrectionTagData struct {
	pd.TFixPointCorrectionTag
}

func (d *FixPointAreaCorrectionTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *FixPointAreaCorrectionTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TFixPointCorrectionTag)
}

type TransformationTagData struct {
	pd.TTransformationTag
}

func (d *TransformationTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *TransformationTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TTransformationTag)
}

type DictionaryTagData struct {
	pd.TDictionaryTagData
}

func (d *DictionaryTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *DictionaryTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TDictionaryTagData)
}

type (
	UserDictionaryTagData = DictionaryTagData
	CarDictionaryTagData  = DictionaryTagData
)

type ClusterTagData struct {
	Threshold uint64
}

func (d *ClusterTagData) UnmarshalJSON(data []byte) error {
	var v struct {
		Threshold uint64 `json:"threshold"`
	}
	if err := json.Unmarshal(data, &v); err != nil {
		return err
	}
	d.Threshold = v.Threshold
	return nil
}

func (d *ClusterTagData) Unmarshal([]byte) error {
	return ErrFormatNotSupported
}

type AreaPotentialTagData struct {
	pf.TAreaPotentialTag
}

func (d *AreaPotentialTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *AreaPotentialTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TAreaPotentialTag)
}

type TelematicsTagData struct {
	pd.TTelematicsTag
}

var commandCodesMap = map[string]uint32{
	"UNKNOWN":                     0,
	"RESTART":                     1,
	"SET_PARAM":                   2,
	"GET_PARAM":                   3,
	"TELEMATIC":                   4,
	"TOFACTORY":                   5,
	"CLEAN_BB":                    6,
	"SET_OUTPUT":                  7,
	"DISCONNECT":                  8,
	"TAKE_PHOTO":                  8,
	"BLINKER_FLASH":               9,
	"CLOSE_DOORS":                 10,
	"OPEN_DOORS":                  11,
	"OPEN_DR_DOOR":                12,
	"HORN":                        13,
	"TRUNK":                       14,
	"STOP_ENGINE":                 15,
	"START_ENGINE":                16,
	"START_WEBASTO":               17,
	"STOP_WEBASTO":                18,
	"EMULATE_DRIVER_DOOR":         19,
	"HORN_AND_BLINK":              20,
	"WINDOWS_CLOSING_3S":          21,
	"WINDOWS_CLOSING_7S":          22,
	"WINDOWS_CLOSING_11S":         23,
	"WINDOWS_CLOSING_15S":         24,
	"WINDOWS_CLOSING_19S":         25,
	"WINDOWS_CLOSING_23S":         26,
	"WINDOWS_CLOSING_29S":         27,
	"WINDOWS_OPENING_3S":          28,
	"WINDOWS_OPENING_7S":          29,
	"WINDOWS_OPENING_11S":         30,
	"WINDOWS_OPENING_15S":         31,
	"WINDOWS_OPENING_19S":         32,
	"WINDOWS_OPENING_23S":         33,
	"WINDOWS_OPENING_29S":         34,
	"START_PROD_TESTS":            35,
	"RESET_LORA_SENS_ALARMS":      36,
	"YADRIVE_WARMING":             37,
	"YADRIVE_STOP_WARMING":        38,
	"YADRIVE_START_OF_LEASE":      39,
	"YADRIVE_END_OF_LEASE":        40,
	"YADRIVE_UNLOCK_HOOD":         41,
	"YADRIVE_LOCK_HOOD":           42,
	"YADRIVE_FORCED_END_OF_LEASE": 43,
	"YADRIVE_PANIC":               44,
	"YADRIVE_AUDIO_COMMAND":       45,
	"ELECTRIC_CAR_COMMAND":        60,
	"AUXILIARY_CAR_COMMAND":       60,
	"NRF_ADD_MARK":                63,
	"NRF_REMOVE_MARK":             64,
	"OBD_FORWARD_CONFIG":          65,
	"OBD_COMMAND":                 66,
	"GSM_MODEM_COMMAND":           68,
	"FAST_DATA_CONFIG":            69,
	// scenario
	"SCENARIO_STOP_WARMING_AND_OPEN_DOORS":             0x80,
	"SCENARIO_STOP_WARMING_AND_END_OF_LEASE":           0x81,
	"SCENARIO_UNLOCK_DOORS_AND_HOOD":                   0x82,
	"SCENARIO_LOCK_DOORS_AND_HOOD":                     0x83,
	"SCENARIO_UNLOCK_HOOD_AND_START_OF_LEASE":          0x84,
	"SCENARIO_STOP_WARMING_END_OF_LEASE_AND_LOCK_HOOD": 0x85,
	"SCENARIO_ENABLE_LONGHORN":                         0x86,
	"SCENARIO_DISABLE_LONGHORN":                        0x87,
	"SCENARIO_POLITE_FORCED_END_OF_LEASE":              0x88,
	"SCENARIO_BLE_RESET":                               0x89,
	"SCENARIO_PREPARE":                                 0x8A,
	"SCENARIO_RESET":                                   0x8B,
	"SCENARIO_FORCED_RESET":                            0x8C,
	"SCENARIO_POLITE_RESTART":                          0x8D,
	"SCENARIO_DIN2_SHUTDOWN":                           0x8E,
	"SCENARIO_POLITE_SET_PARAM":                        0x8F,
	// car control for TTelematicsTestClient
	"MOVE_TO_COORD": 0xf0,
}

var commandCodesReverseMap = map[uint32]string{}

func init() {
	for key, value := range commandCodesMap {
		commandCodesReverseMap[value] = key
	}
}

func (d *TelematicsTagData) UnmarshalJSON(data []byte) error {
	var v struct {
		Command string `json:"command"`
		Handler string `json:"handler"`
	}
	if err := json.Unmarshal(data, &v); err != nil {
		return err
	}
	command := commandCodesMap[v.Command]
	d.Command = &command
	d.Handler = &v.Handler
	return nil
}

func (d *TelematicsTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TTelematicsTag)
}

func (d *TelematicsTagData) MarshalJSON() ([]byte, error) {
	var v struct {
		Command string `json:"command"`
		Handler string `json:"handler"`
	}
	v.Command = commandCodesReverseMap[0]
	if d.Command != nil {
		if command, ok := commandCodesReverseMap[*d.Command]; ok {
			v.Command = command
		}
	}
	if d.Handler != nil {
		v.Handler = *d.Handler
	}
	return json.Marshal(v)
}

type DeviceSnapshot struct {
	pf.THistoryDeviceSnapshot
}

func (d *DeviceSnapshot) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *DeviceSnapshot) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.THistoryDeviceSnapshot)
}

type CarServiceTagData = SimpleUserTagData

type PhotoRequestTagData = SimpleUserTagData

type RepairTagData struct {
	pd.TMajorTagData
}

func (d *RepairTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *RepairTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TMajorTagData)
}

func getBool(v interface{}) (bool, bool) {
	switch t := v.(type) {
	case bool:
		return t, true
	case int64:
		if t == 1 {
			return true, true
		}
		return false, true
	default:
		return false, false
	}
}

func getString(v interface{}) (string, bool) {
	switch t := v.(type) {
	case string:
		return t, true
	case []byte:
		return string(t), true
	default:
		return "", false
	}
}

func getInt64(v interface{}) (int64, bool) {
	switch t := v.(type) {
	case int64:
		return t, true
	case uint64:
		return int64(t), true
	default:
		return 0, false
	}
}

func getUInt64(v interface{}) (uint64, bool) {
	switch t := v.(type) {
	case uint64:
		return t, true
	case int64:
		return uint64(t), true
	default:
		return 0, false
	}
}

func getInt(v interface{}) (int, bool) {
	t, ok := getUInt64(v)
	return int(t), ok
}

func getUInt16(v interface{}) (uint16, bool) {
	t, ok := getUInt64(v)
	return uint16(t), ok
}

func getFloat64(v interface{}) (float64, bool) {
	switch t := v.(type) {
	case float64:
		return t, true
	case int64:
		return float64(t), true
	default:
		return 0, false
	}
}

func getFloat32(v interface{}) (float32, bool) {
	t, ok := getFloat64(v)
	return float32(t), ok
}

type TelematicsConfigurationTagData struct {
	ApplyOnAdd                     bool
	ConfigurationHash              uint64
	Deadline                       int64
	AuxFuelLevelTransmissionPeriod uint16
	MaxAuxFuelVolume               [10]float32
}

func (d *TelematicsConfigurationTagData) UnmarshalJSON(data []byte) error {
	var v map[string]interface{}
	if err := json.Unmarshal(data, &v); err != nil {
		return err
	}
	d.ApplyOnAdd, _ = getBool(v["apply_on_add"])
	d.ConfigurationHash, _ = getUInt64(v["sensors_hash"])
	d.Deadline, _ = getInt64(v["deadline"])
	d.AuxFuelLevelTransmissionPeriod, _ = getUInt16(
		v["aux_fuel_level_transmission_period"],
	)
	for i := 0; i < len(d.MaxAuxFuelVolume); i++ {
		d.MaxAuxFuelVolume[i], _ = getFloat32(v[d.maxAuxFuelVolumeName(i)])
	}
	return nil
}

func (d *TelematicsConfigurationTagData) Unmarshal([]byte) error {
	return ErrFormatNotSupported
}

func (d TelematicsConfigurationTagData) maxAuxFuelVolumeName(i int) string {
	return fmt.Sprintf("max_aux_fuel_volume_%d", i)
}

type TelematicsFirmwareTagData struct {
	Alias    string
	FullName string
	Model    string
	Version  string
	Tag      string
	Revision uint32
	Type     int
}

func (d *TelematicsFirmwareTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *TelematicsFirmwareTagData) Unmarshal(data []byte) error {
	return ErrFormatNotSupported
}

type TelematicsDeferredCommandTagData struct {
	Since    int64
	Until    int64
	Duration int
}

func (d *TelematicsDeferredCommandTagData) UnmarshalJSON(data []byte) error {
	var v map[string]interface{}
	if err := json.Unmarshal(data, &v); err != nil {
		return err
	}
	d.Since, _ = getInt64(v["since"])
	d.Until, _ = getInt64(v["until"])
	d.Duration, _ = getInt(v["duration"])
	return nil
}

func (d *TelematicsDeferredCommandTagData) Unmarshal(data []byte) error {
	return d.UnmarshalJSON(data)
}

type SimpleCarTagData = SimpleUserTagData

type UserOriginTagData struct {
	Origin         string
	ExternalUserID string
}

func (d *UserOriginTagData) UnmarshalJSON(data []byte) error {
	var v map[string]interface{}
	if err := json.Unmarshal(data, &v); err != nil {
		return err
	}
	d.Origin, _ = getString(v["origin"])
	d.ExternalUserID, _ = getString(v["external_user_id"])
	return nil
}

func (d *UserOriginTagData) Unmarshal(data []byte) error {
	return d.UnmarshalJSON(data)
}

type CarScannerTagData struct {
	carscannerpb.TScannerTag
}

func (d *CarScannerTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *CarScannerTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TScannerTag)
}

type ScoringTagData struct {
	pf.TScoringTag
}

func (d *ScoringTagData) UnmarshalJSON([]byte) error {
	return ErrFormatNotSupported
}

func (d *ScoringTagData) Unmarshal(data []byte) error {
	return proto.Unmarshal(data, &d.TScoringTag)
}
