package models

import (
	"context"
	"database/sql"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/zootopia/analytics/drive/models"
	"a.yandex-team.ru/zootopia/library/go/db"
)

type UserTagStore struct {
	*baseStore
}

// Store returns direct store without cache.
func (s *UserTagStore) Store() *UserTagStore {
	return s
}

func (s *UserTagStore) FindByUserTx(
	tx *sql.Tx, userID uuid.UUID,
) ([]models.Tag, error) {
	var tags []models.Tag
	rows, err := s.objects.FindObjects(
		tx, `"object_id" = $1`, userID,
	)
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = rows.Close()
	}()
	for rows.Next() {
		tags = append(tags, rows.Object().(models.Tag))
	}
	return tags, rows.Err()
}

func (s *UserTagStore) FindByUser(userID uuid.UUID) ([]models.Tag, error) {
	var tags []models.Tag
	if err := gosql.WithTxContext(
		context.Background(), s.db, &sql.TxOptions{ReadOnly: true},
		func(tx *sql.Tx) (err error) {
			tags, err = s.FindByUserTx(tx, userID)
			return
		},
	); err != nil {
		return nil, err
	}
	return tags, nil
}

type CachedUserTagStore struct {
	*UserTagStore
	*baseCachedStore
}

func NewUserTagStore(conn *sql.DB) *UserTagStore {
	return &UserTagStore{
		baseStore: newBaseStore(
			conn, db.Postgres,
			models.Tag{}, "tag_id", "user_tags",
			models.TagEvent{}, "history_event_id", "user_tags_history",
		),
	}
}

func NewCachedUserTagStore(
	store *UserTagStore, cache *sql.DB,
) *CachedUserTagStore {
	initFunc := func(tx *sql.Tx) error {
		_, err := tx.Exec(
			`DROP TABLE "user_tag_cache" IF EXISTS;` +
				`CREATE TABLE "user_tag_cache" (` +
				`)`,
		)
		return err
	}
	return &CachedUserTagStore{
		UserTagStore:    store,
		baseCachedStore: newBaseCachedStore(store.baseStore, cache, initFunc),
	}
}
