package core

import (
	"context"
	"database/sql"
	"fmt"
	"net/http"
	"os"
	"strings"
	"sync"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/analytics/gobase/config"
	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/library/go/auth"
	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/drive/library/go/solomon"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm"
	tp2 "a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
	"a.yandex-team.ru/zootopia/analytics/drive/api"
	"a.yandex-team.ru/zootopia/library/go/db"

	dm "a.yandex-team.ru/zootopia/analytics/drive/models"
)

type Core struct {
	// Config contains drive config.
	Config *config.Config
	// DBs contains database connections.
	DBs map[string]*gosql.DB
	// TVMs contains TVM connections.
	TVMs map[string]tvm.Client
	// YTs contains YT connections.
	YTs map[string]yt.Client
	// drives contains Drive clients.
	Drives map[string]*api.Client
	// Locks contains lock store.
	Locks *models.LockStore
	// States contains states store.
	States *models.StateStore
	// DriveDB contains drive DB connection.
	DriveDB *sql.DB
	// Drive contains drive API client.
	Drive *api.Client
	// DriveUsers contains drive users store.
	DriveUsers *dm.UserStore
	// SolomonOld contains.
	SolomonOld *solomon.PushClient
	// Solomon contains client for solomon.
	Solomon *solomon.Client
	// BB contains Blackbox client.
	BB blackbox.Client
	// YT contains YT client.
	YT yt.Client
	// logger contains logger.
	logger log.Logger
	//
	context context.Context
	cancel  context.CancelFunc
	waiter  sync.WaitGroup
}

// Start starts core.
func (c *Core) Start() error {
	if c.cancel != nil {
		return fmt.Errorf("core already started")
	}
	c.context, c.cancel = context.WithCancel(context.Background())
	if c.SolomonOld != nil {
		c.SolomonOld.Start()
	}
	return nil
}

// Stop stops core.
func (c *Core) Stop() {
	if c.cancel == nil {
		return
	}
	c.cancel()
	c.waiter.Wait()
	if c.Solomon != nil {
		c.Solomon.Close()
	}
	if c.SolomonOld != nil {
		c.SolomonOld.Stop()
	}
}

// GetServiceTicket returns auth with ServiceTicket.
func (c *Core) GetServiceTicket(
	source, target string,
) (auth.ServiceTicket, error) {
	tvm, ok := c.TVMs[source]
	if !ok {
		return auth.ServiceTicket{}, fmt.Errorf(
			"tvm client %q not configured", source,
		)
	}
	return auth.ServiceTicket{TVM: tvm, Target: target}, nil
}

// Signal returns signal wrapper for solomon monitoring.
func (c *Core) Signal(name string, tags map[string]string) *solomon.Signal {
	if c.Solomon == nil {
		return solomon.StubSignal
	}
	return c.Solomon.Signal(name, tags)
}

// SignalV sends signal to solomon monitoring.
func (c *Core) SignalV(
	sensor string, value interface{}, options ...solomon.SignalOption,
) {
	if c.SolomonOld != nil {
		c.SolomonOld.Signal(sensor, value, options...)
	}
}

// StartTask starts function in new goroutine.
//
// Context passed to function will be cancelled when core is stopped.
// Note that task function should recover panics when it is possible,
// otherwise core will fall down.
func (c *Core) StartTask(task func(ctx context.Context)) {
	c.waiter.Add(1)
	go func() {
		defer c.waiter.Done()
		task(c.context)
	}()
}

// StartDaemon starts function in new goroutine.
//
// Context passed to function will be cancelled when core is stopped.
func (c *Core) StartDaemon(task func(ctx context.Context) error) {
	c.StartTask(func(ctx context.Context) {
		ticker := time.NewTicker(5 * time.Second)
		defer ticker.Stop()
		if err := task(ctx); err != nil {
			c.logger.Error("Daemon task failed", log.Error(err))
		}
		for {
			select {
			case <-ctx.Done():
				return
			case <-ticker.C:
				// Kostyl for random select.
				select {
				case <-ctx.Done():
					return
				default:
				}
				if err := task(ctx); err != nil {
					c.logger.Error("Daemon task failed", log.Error(err))
				}
			}
		}
	})
}

// StartClusterDaemon starts infinite loop of task function calls.
//
// Its guaranteed that only one task in whole cluster will be running
// at the same time. Note, that task function should finish its work
// as soon as possible when context is cancelled.
func (c *Core) StartClusterDaemon(
	name string, task func(ctx context.Context) error,
) {
	c.StartTask(func(ctx context.Context) {
		ticker := time.NewTicker(5 * time.Second)
		defer ticker.Stop()
		if err := c.Locks.WithLock(ctx, name, task); badError(err) {
			c.logger.Error("Daemon task failed", log.Error(err))
		}
		for {
			select {
			case <-ctx.Done():
				return
			case <-ticker.C:
				// Kostyl for random select.
				select {
				case <-ctx.Done():
					return
				default:
				}
				if err := c.Locks.WithLock(ctx, name, task); badError(err) {
					c.logger.Error("Daemon task failed", log.Error(err))
				}
			}
		}
	})
}

func badError(err error) bool {
	return err != nil && err != models.ErrLockAcquired &&
		err != models.ErrLockReleased && err != context.Canceled
}

const (
	AuthYandexUser   = "auth_yandex_user"
	AuthDriveUser    = "auth_drive_user"
	AuthDriveActions = "auth_drive_actions"
	AuthDriveRoles   = "auth_drive_roles"
)

type YandexUser struct {
	ID    uint64
	Login string
}

type DriveUser = dm.User

type DriveRoles map[string]struct{}

type DriveActions map[string]struct{}

func getClientIP(c echo.Context) string {
	if ip := c.Request().Header.Get("X-Forwarded-For-Y"); ip != "" {
		return ip
	}
	return c.RealIP()
}

func getHost(host string) string {
	if len(host) > 0 && host[len(host)-1] != ']' {
		if pos := strings.LastIndex(host, ":"); pos != -1 {
			return host[:pos]
		}
	}
	return host
}

func (c *Core) ExtractYandexUser(next echo.HandlerFunc) echo.HandlerFunc {
	return func(ctx echo.Context) error {
		sessionID, err := ctx.Cookie("Session_id")
		if err != nil {
			return next(ctx)
		}
		info, err := c.BB.SessionID(
			ctx.Request().Context(),
			blackbox.SessionIDRequest{
				SessionID:     sessionID.Value,
				UserIP:        getClientIP(ctx),
				Host:          getHost(ctx.Request().Host),
				GetUserTicket: false,
			},
		)
		if err != nil {
			ctx.Logger().Error(err)
			return next(ctx)
		}
		ctx.Set(AuthYandexUser, YandexUser{
			ID:    info.User.ID,
			Login: info.User.Login,
		})
		return next(ctx)
	}
}

func (c *Core) ExtractDriveUser(next echo.HandlerFunc) echo.HandlerFunc {
	return func(ctx echo.Context) error {
		yandexUser, ok := ctx.Get(AuthYandexUser).(YandexUser)
		if !ok {
			// Yandex user does not found.
			return next(ctx)
		}
		driveUser, err := c.DriveUsers.GetByUID(yandexUser.ID)
		if err != nil {
			ctx.Logger().Error(err)
			if err != sql.ErrNoRows {
				return err
			}
			return next(ctx)
		}
		ctx.Set(AuthDriveUser, driveUser)
		return next(ctx)
	}
}

func (c *Core) ExtractDriveActions(next echo.HandlerFunc) echo.HandlerFunc {
	return func(ctx echo.Context) error {
		driveUser, ok := ctx.Get(AuthDriveUser).(DriveUser)
		if !ok {
			// Drive user does not found.
			return next(ctx)
		}
		actions, err := c.Drive.GetUserActions(driveUser.ID.String())
		if err != nil {
			ctx.Logger().Error(err)
			return next(ctx)
		}
		driveActions := DriveActions{}
		for _, action := range actions {
			if action.Enabled && action.UserID == driveUser.ID.String() {
				driveActions[action.ID] = struct{}{}
			}
		}
		ctx.Set(AuthDriveActions, driveActions)
		return next(ctx)
	}
}

func (c *Core) ExtractDriveRoles(next echo.HandlerFunc) echo.HandlerFunc {
	return func(ctx echo.Context) error {
		driveUser, ok := ctx.Get(AuthDriveUser).(DriveUser)
		if !ok {
			// Drive user does not found.
			return next(ctx)
		}
		roles, err := c.Drive.GetUserRoles(driveUser.ID.String())
		if err != nil {
			ctx.Logger().Error(err)
			return next(ctx)
		}
		driveRoles := DriveRoles{}
		for _, role := range roles {
			if role.Active && role.UserID == driveUser.ID.String() {
				driveRoles[role.RoleID] = struct{}{}
			}
		}
		ctx.Set(AuthDriveRoles, driveRoles)
		return next(ctx)
	}
}

type ErrorResponse struct {
	Message             string   `json:"message"`
	MissingDriveActions []string `json:"missing_drive_actions,omitempty"`
	MissingDriveRoles   []string `json:"missing_drive_roles,omitempty"`
}

func (c *Core) RequireDriveAction(action string) echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(ctx echo.Context) error {
			driveActions, ok := ctx.Get(AuthDriveActions).(DriveActions)
			if !ok {
				return ctx.JSON(http.StatusUnauthorized, ErrorResponse{
					Message: "drive actions are not fetched",
				})
			}
			if _, ok := driveActions[action]; !ok {
				return ctx.JSON(http.StatusForbidden, ErrorResponse{
					Message:             "account has no such permissions",
					MissingDriveActions: []string{action},
				})
			}
			return next(ctx)
		}
	}
}

func (c *Core) RequireDriveRole(role string) echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(ctx echo.Context) error {
			driveRoles, ok := ctx.Get(AuthDriveRoles).(DriveRoles)
			if !ok {
				return ctx.JSON(http.StatusUnauthorized, ErrorResponse{
					Message: "drive roles are not fetched",
				})
			}
			if _, ok := driveRoles[role]; !ok {
				return ctx.JSON(http.StatusForbidden, ErrorResponse{
					Message:           "account has no such permissions",
					MissingDriveRoles: []string{role},
				})
			}
			return next(ctx)
		}
	}
}

// Logger returns logger.
func (c *Core) Logger(name string) log.Logger {
	if name == "" {
		return c.logger
	}
	return c.logger.WithName(name)
}

type Option func(*Core) error

func WithLogger(logger log.Logger) Option {
	return func(c *Core) error {
		c.logger = logger
		return nil
	}
}

// NewCore creates a new instance of Core.
func NewCore(cfg *config.Config, options ...Option) (*Core, error) {
	c := Core{Config: cfg}
	for _, option := range options {
		if err := option(&c); err != nil {
			return nil, err
		}
	}
	if c.logger == nil {
		logger, err := zap.NewDeployLogger(cfg.LogLevel)
		if err != nil {
			return nil, err
		}
		c.logger = logger
	}
	var err error
	if cfg := c.Config.DBs; cfg != nil {
		if c.DBs, err = models.SetupDBs(cfg); err != nil {
			return nil, err
		}
	}
	if cfg := c.Config.TVMs; cfg != nil {
		if c.TVMs, err = setupTVMs(cfg, c.logger); err != nil {
			return nil, err
		}
	}
	if cfg := c.Config.YTs; cfg != nil {
		if c.YTs, err = setupYTs(cfg, c.logger); err != nil {
			return nil, err
		}
	}
	if cfg := c.Config.Solomon; cfg != nil {
		tvmAuth, err := c.GetServiceTicket(cfg.Source, cfg.Target)
		if err != nil {
			return nil, err
		}
		hostname, err := os.Hostname()
		if err != nil {
			return nil, err
		}
		c.Solomon, err = solomon.NewClient(
			cfg.Project, cfg.Cluster, cfg.Service,
			solomon.WithAuth(tvmAuth),
			solomon.WithTags(map[string]string{"host": hostname}),
		)
		if err != nil {
			return nil, err
		}
		c.SolomonOld = solomon.NewPushClient(solomon.PushConfig{
			Endpoint: cfg.Endpoint,
			Project:  cfg.Project,
			Cluster:  cfg.Cluster,
			Service:  cfg.Service,
		}, tvmAuth)
	}
	c.Drives = map[string]*api.Client{}
	for name, cfg := range c.Config.Drives {
		c.Drives[name] = api.New(api.Config{
			Endpoint: cfg.Endpoint,
			Token:    cfg.Token.Secret(),
			Query:    cfg.Query,
		})
	}
	if cfg := c.Config.Drive; cfg != nil {
		c.Drive = api.New(api.Config{
			Endpoint: cfg.Endpoint,
			Token:    cfg.Token.Secret(),
			Query:    cfg.Query,
		})
		c.Drives[""] = c.Drive
	}
	if cfg := c.Config.YT; cfg != nil {
		yc, err := ythttp.NewClient(&yt.Config{
			Proxy:  cfg.Proxy,
			Token:  cfg.Token.Secret(),
			Logger: c.logger.Structured(),
		})
		if err != nil {
			return nil, err
		}
		c.YT = yc
	}
	if name := c.Config.CoreDB; name != "" {
		conn, ok := c.DBs[name]
		if !ok {
			return nil, fmt.Errorf("db %q does not exists", name)
		}
		c.Locks = models.NewLockStore(conn, "lock", c.logger)
		c.States = models.NewStateStore(conn, "state")
	}
	if name := c.Config.BackendDB; name != "" {
		conn, ok := c.DBs[name]
		if !ok {
			return nil, fmt.Errorf("db %q does not exists", name)
		}
		c.DriveDB = conn.DB
		c.DriveUsers = dm.NewUserStore(db.Postgres, conn.DB)
	}
	if cfg := c.Config.BB; cfg != nil {
		tvm2, ok := c.TVMs[cfg.Source]
		if !ok {
			return nil, fmt.Errorf("tvm %q does not exists", cfg.Source)
		}
		bbEnv, err := getBlackboxEnv(cfg.Env)
		if err != nil {
			return nil, err
		}
		c.BB, err = httpbb.NewClient(bbEnv, httpbb.WithTVM(tvm2))
		if err != nil {
			return nil, err
		}
	}
	return &c, nil
}

func setupTVMs(
	tvmsCfg map[string]config.TVM, logger log.Logger,
) (map[string]tvm.Client, error) {
	tvms := map[string]tvm.Client{}
	for name, tvmCfg := range tvmsCfg {
		settings := tp2.TvmAPISettings{
			SelfID: tvmCfg.Source,
			ServiceTicketOptions: tp2.NewAliasesOptions(
				tvmCfg.Secret.Secret(), tvmCfg.Targets,
			),
			EnableServiceTicketChecking: true,
		}
		var err error
		tvms[name], err = tp2.NewAPIClient(settings, logger)
		if err != nil {
			return nil, err
		}
	}
	return tvms, nil
}

func setupYTs(
	ytsCfg map[string]config.YT, logger log.Logger,
) (map[string]yt.Client, error) {
	yts := map[string]yt.Client{}
	for name, ytCfg := range ytsCfg {
		config := yt.Config{
			Proxy:  ytCfg.Proxy,
			Token:  ytCfg.Token.Secret(),
			Logger: logger.Structured(),
		}
		var err error
		yts[name], err = ythttp.NewClient(&config)
		if err != nil {
			return nil, err
		}
	}
	return yts, nil
}

func getBlackboxEnv(name string) (httpbb.Environment, error) {
	switch name {
	case "intranet", "":
		return httpbb.IntranetEnvironment, nil
	case "test":
		return httpbb.TestEnvironment, nil
	case "prod":
		return httpbb.ProdEnvironment, nil
	case "mimino":
		return httpbb.MiminoEnvironment, nil
	}
	return httpbb.Environment{}, fmt.Errorf(
		"unsupported Blackbox environment: %q", name,
	)
}
