package models

import (
	"database/sql/driver"
	"encoding/json"
	"fmt"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/yt/go/yson"
)

// JSON represents raw JSON value.
type JSON []byte

// Clone creates copy of JSON object.
func (v JSON) Clone() JSON {
	c := make(JSON, len(v))
	copy(c, v)
	return c
}

// MarshalJSON mashals JSON.
func (v JSON) MarshalJSON() ([]byte, error) {
	if len(v) == 0 {
		return []byte("null"), nil
	}
	if !json.Valid(v) {
		return nil, fmt.Errorf("invalid json value")
	}
	return v, nil
}

func (v JSON) MarshalYSON() ([]byte, error) {
	var d interface{}
	if err := json.Unmarshal(v, &d); err != nil {
		return nil, err
	}
	return yson.Marshal(d)
}

// UnmarshalJSON unmarshals JSON.
func (v *JSON) UnmarshalJSON(bytes []byte) error {
	if len(bytes) == 0 {
		*v = nil
		return nil
	}
	if !json.Valid(bytes) {
		return fmt.Errorf("invalid json value")
	}
	*v = JSON(bytes)
	return nil
}

// Value returns SQL value.
func (v JSON) Value() (driver.Value, error) {
	value, err := v.MarshalJSON()
	if err != nil {
		return nil, err
	}
	return string(value), nil
}

// Scan scans SQL value.
func (v *JSON) Scan(value interface{}) error {
	switch x := value.(type) {
	case nil:
		*v = nil
		return nil
	case []byte:
		return v.UnmarshalJSON(x)
	case string:
		return v.UnmarshalJSON([]byte(x))
	default:
		return fmt.Errorf("unsupported value: %T", x)
	}
}

type NBool bool

func (v NBool) Value() (driver.Value, error) {
	return bool(v), nil
}

func (v *NBool) Scan(value interface{}) error {
	switch x := value.(type) {
	case nil:
		*v = false
	case bool:
		*v = NBool(x)
	default:
		return fmt.Errorf("unsupported value: %T", x)
	}
	return nil
}

func (v NBool) MarshalYSON() ([]byte, error) {
	return yson.Marshal(bool(v))
}

func (v *NBool) UnmarshalYSON(data []byte) error {
	var value *bool
	if err := yson.Unmarshal(data, &value); err != nil {
		return err
	}
	if value == nil {
		*v = false
	} else {
		*v = NBool(*value)
	}
	return nil
}

type NInt64 int64

func (v NInt64) Value() (driver.Value, error) {
	if v == 0 {
		return nil, nil
	}
	return int64(v), nil
}

func (v *NInt64) Scan(value interface{}) error {
	switch x := value.(type) {
	case nil:
		*v = 0
	case int64:
		*v = NInt64(x)
	default:
		return fmt.Errorf("unsupported value: %T", x)
	}
	return nil
}

func (v *NInt64) UnmarshalYSON(data []byte) error {
	var value *int64
	if err := yson.Unmarshal(data, &value); err != nil {
		return err
	}
	if value == nil {
		*v = 0
	} else {
		*v = NInt64(*value)
	}
	return nil
}

type NTime struct {
	time.Time
}

func (v NTime) Value() (driver.Value, error) {
	if v.IsZero() {
		return nil, nil
	}
	return driver.Value(v.Time), nil
}

func (v *NTime) Scan(value interface{}) error {
	switch x := value.(type) {
	case nil:
		*v = NTime{}
	case time.Time:
		v.Time = x
	default:
		return fmt.Errorf("unsupported value: %T", x)
	}
	return nil
}

type NUUID struct {
	uuid.UUID
}

func (v NUUID) Value() (driver.Value, error) {
	if v.UUID == uuid.Nil {
		return nil, nil
	}
	return v.UUID.Value()
}

func (v *NUUID) Scan(value interface{}) error {
	switch value.(type) {
	case nil:
		v.UUID = uuid.Nil
	default:
		return v.UUID.Scan(value)
	}
	return nil
}
