package models

import (
	"database/sql"
	"encoding/json"

	"a.yandex-team.ru/drive/library/go/gosql"
)

type State struct {
	ID    int64  `db:"id"`
	Name  string `db:"name"`
	State JSON   `db:"state"`
}

// Clone creates copy of State.
func (s State) Clone() State {
	s.State = s.State.Clone()
	return s
}

func (s *State) ScanState(v interface{}) error {
	return json.Unmarshal(s.State, &v)
}

func (s *State) SetState(v interface{}) error {
	var err error
	s.State, err = json.Marshal(v)
	return err
}

type StateStore struct {
	db    *gosql.DB
	table string
}

// DB returns database that used by store.
func (s *StateStore) DB() *gosql.DB {
	return s.db
}

func (s *StateStore) GetByNameTx(tx gosql.Runner, name string) (State, error) {
	var state State
	names, scans := gosql.StructNameValues(&state, true)
	query, values := s.db.Select(s.table).
		Names(names...).Where(gosql.Column("name").Equal(name)).Build()
	row := tx.QueryRow(query, values...)
	if err := row.Scan(scans...); err != nil {
		return State{}, err
	}
	return state, nil
}

func (s *StateStore) GetOrCreateByNameTx(tx gosql.Runner, name string) (State, error) {
	state, err := s.GetByNameTx(tx, name)
	if err != nil {
		if err != sql.ErrNoRows {
			return State{}, err
		}
		state.Name = name
		if err := s.CreateTx(tx, &state); err != nil {
			return State{}, err
		}
	}
	return state, nil
}

func (s *StateStore) GetOrCreateByName(name string) (State, error) {
	return s.GetOrCreateByNameTx(s.db, name)
}

func (s *StateStore) GetLockedByNameTx(tx gosql.Runner, name string) (State, error) {
	var state State
	names, scans := gosql.StructNameValues(&state, true)
	query, values := s.db.Select(s.table).
		Names(names...).Where(gosql.Column("name").Equal(name)).
		ForUpdate(true).Build()
	row := tx.QueryRow(query, values...)
	if err := row.Scan(scans...); err != nil {
		return State{}, err
	}
	return state, nil
}

func (s *StateStore) CreateTx(tx gosql.Runner, state *State) error {
	names, values := gosql.StructNameValues(state, false, "id")
	query, values := s.db.Insert(s.table).
		Names(names...).Values(values...).Build()
	row := tx.QueryRow(query+` RETURNING "id"`, values...)
	if len(state.State) == 0 {
		state.State = JSON("null")
	}
	return row.Scan(&state.ID)
}

func (s *StateStore) UpdateTx(tx gosql.Runner, state State) error {
	names, values := gosql.StructNameValues(state, false, "id")
	query, values := s.db.Update(s.table).
		Names(names...).Values(values...).
		Where(gosql.Column("id").Equal(state.ID)).Build()
	if res, err := tx.Exec(query, values...); err != nil {
		return err
	} else if rows, err := res.RowsAffected(); err != nil {
		return err
	} else if rows != 1 {
		return sql.ErrNoRows
	}
	return nil
}

func (s *StateStore) Update(state State) error {
	return s.UpdateTx(s.db, state)
}

func NewStateStore(db *gosql.DB, table string) *StateStore {
	return &StateStore{db: db, table: table}
}
