package aggression

import (
	"database/sql"
	"fmt"
	"sync"
	"time"

	"github.com/gofrs/uuid"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gobase/models"
	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/zootopia/analytics/drive/api"
)

func init() {
	updateCarsCmd := cobra.Command{
		Use: "update-cars",
		Run: gotasks.WrapMain(updateCarsMain),
	}
	updateCarsCmd.Flags().Bool("dry-run", false, "Enables dry-run mode")
	updateCarsCmd.PersistentFlags().String("yt-proxy", "hahn", "YT proxy")
	updateCarsCmd.Flags().String("drive", "", "Name of Drive client")
	updateCarsCmd.PersistentFlags().String("db", "analytics", "DB name")
	AggressionCmd.AddCommand(&updateCarsCmd)
}

type carStateData struct {
	// Score contains car score.
	Score float64 `json:"score"`
	// Rank contains car rank.
	Rank float64 `json:"rank"`
	// ScoreTime contains time of score.
	ScoreTime int64 `json:"score_time"`
}

type carScoringRow struct {
	Time  int64     `yson:"timestamp"`
	CarID uuid.UUID `yson:"user_id"`
	Score float64   `yson:"score"`
	Rank  float64   `yson:"rank"`
}

func processCarScoringRow(
	ctx *gotasks.Context,
	db *gosql.DB,
	drive *api.Client,
	row carScoringRow,
	dryRun bool,
) error {
	if err := gosql.WithTxContext(
		ctx.Context, db, nil, func(tx *sql.Tx) error {
			carRawState, err := ctx.States.GetLockedByNameTx(
				tx, "aggression/cars/states/"+row.CarID.String(),
			)
			if err != nil {
				if err != sql.ErrNoRows {
					return err
				}
				carRawState.Name = "aggression/cars/states/" + row.CarID.String()
				if err := ctx.States.CreateTx(tx, &carRawState); err != nil {
					ctx.Signal("aggression.cars.create_car_state.error_sum", nil).Add(1)
					return err
				}
			}
			carState := carStateData{}
			if err := carRawState.ScanState(&carState); err != nil {
				return err
			}
			if row.Time <= carState.ScoreTime {
				return nil
			}
			scoreData := api.GenericTagData{
				"value":     row.Score,
				"rank":      row.Rank,
				"timestamp": row.Time,
			}
			if carState.Score > 0 {
				scoreData["previous_value"] = carState.Score
			}
			carState.Score = row.Score
			carState.Rank = row.Rank
			carState.ScoreTime = row.Time
			scoreTag := api.CarTag{
				Tag:   "scoring_car_tag",
				Data:  &scoreData,
				CarID: row.CarID.String(),
			}
			ctx.Logger.Debug(
				"Updating car tag",
				log.Any("tag", scoreTag),
			)
			if !dryRun {
				if err := drive.AddCarTag(scoreTag); err != nil {
					ctx.Signal("aggression.cars.update_scoring_car_tag.error_sum", nil).Add(1)
					return err
				}
				ctx.Signal("aggression.cars.update_scoring_car_tag.ok_sum", nil).Add(1)
			}
			if err := carRawState.SetState(carState); err != nil {
				return err
			}
			if !dryRun {
				if err := ctx.States.UpdateTx(tx, carRawState); err != nil {
					ctx.Signal("aggression.cars.update_car_state.error_sum", nil).Add(1)
					return err
				}
				ctx.Signal("aggression.cars.update_car_state.ok_sum", nil).Add(1)
			}
			return nil
		}); err != nil {
		return err
	}
	return nil
}

func updateCarReaderState(
	ctx *gotasks.Context,
	db *gosql.DB,
	rawState *models.State,
	state *readerStateData,
	lastTime int64,
	dryRun bool,
) error {
	ctx.Logger.Info(
		"Update reader state",
		log.Int64("last_time", lastTime),
		log.Bool("dryrun", dryRun),
	)
	if lastTime < state.LastTime {
		ctx.Signal("aggression.cars.update_state.error_sum", nil).Add(1)
		return fmt.Errorf(
			"last time is too small: %d < %d",
			lastTime, state.LastTime,
		)
	}
	state.LastTime = lastTime
	if err := rawState.SetState(state); err != nil {
		ctx.Signal("aggression.cars.update_state.error_sum", nil).Add(1)
		return err
	}
	if !dryRun {
		if err := ctx.States.UpdateTx(db, *rawState); err != nil {
			ctx.Signal("aggression.cars.update_state.error_sum", nil).Add(1)
			return err
		}
		ctx.Signal("aggression.cars.update_state.ok_sum", nil).Add(1)
	}
	return nil
}

func processCarScoringRows(
	ctx *gotasks.Context,
	db *gosql.DB,
	yc yt.Client,
	drive *api.Client,
	state readerStateData,
	table ypath.Path,
	dryRun bool,
) (lastTime int64, err error) {
	ctx.Logger.Info(
		"Reading cars",
		log.Int64("last_time", state.LastTime),
	)
	var timesMutex sync.Mutex
	timesCount := map[int64]int{}
	lastTime = state.LastTime
	defer func() {
		for time, count := range timesCount {
			if count > 0 && time < lastTime {
				lastTime = time
			}
		}
	}()
	var waiter sync.WaitGroup
	defer waiter.Wait()
	rows := make(chan carScoringRow)
	defer close(rows)
	for i := 0; i < 32; i++ {
		waiter.Add(1)
		go func() {
			defer waiter.Done()
			for row := range rows {
				if err := processCarScoringRow(
					ctx, db, drive, row, dryRun,
				); err != nil {
					ctx.Logger.Error(
						"Unable to process car scoring row",
						log.Any("row", row),
						log.Error(err),
					)
				} else {
					func() {
						timesMutex.Lock()
						defer timesMutex.Unlock()
						timesCount[row.Time]--
					}()
				}
			}
		}()
	}
	in, err := yc.ReadTable(
		ctx.Context,
		table.Rich().AddRange(
			ypath.StartingFrom(ypath.Key(state.LastTime)),
		),
		nil,
	)
	if err != nil {
		return 0, err
	}
	defer func() {
		_ = in.Close()
	}()
	for in.Next() {
		select {
		case <-ctx.Context.Done():
			err = ctx.Context.Err()
			return
		default:
		}
		var row carScoringRow
		if err = in.Scan(&row); err != nil {
			return
		}
		if row.Time < state.LastTime {
			err = fmt.Errorf("invalid update time of record")
			return
		}
		if row.Time > lastTime {
			lastTime = row.Time
		}
		func() {
			timesMutex.Lock()
			defer timesMutex.Unlock()
			timesCount[row.Time]++
		}()
		select {
		case rows <- row:
		case <-ctx.Context.Done():
			err = ctx.Context.Err()
			return
		}
	}
	err = in.Err()
	return
}

func updateCarsMain(ctx *gotasks.Context) error {
	dryRun, err := ctx.Cmd.Flags().GetBool("dry-run")
	if err != nil {
		return err
	}
	yc, err := ctx.GetYT()
	if err != nil {
		return err
	}
	dbName, err := ctx.Cmd.Flags().GetString("db")
	if err != nil {
		return err
	}
	db, ok := ctx.DBs[dbName]
	if !ok {
		return fmt.Errorf("invalid DB name %q", dbName)
	}
	driveName, err := ctx.Cmd.Flags().GetString("drive")
	if err != nil {
		return err
	}
	drive, ok := ctx.Drives[driveName]
	if !ok {
		return fmt.Errorf("invalid Drive name %q", driveName)
	}
	rawReaderState, err := ctx.States.GetOrCreateByNameTx(db, "aggression/cars/reader")
	if err != nil {
		return err
	}
	readerState := readerStateData{}
	if err := rawReaderState.ScanState(&readerState); err != nil {
		return err
	}
	lastTime, err := processCarScoringRows(
		ctx, db, yc, drive, readerState,
		ctx.Config.YTPaths.AggressionCarScoreTable,
		dryRun,
	)
	if err != nil {
		return err
	}
	ctx.Signal("aggression.cars.update_lag_last", nil).
		Set(float64(time.Now().Unix() - lastTime))
	return updateCarReaderState(
		ctx, db, &rawReaderState, &readerState, lastTime, dryRun,
	)
}
