package cars

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"sort"
	"strconv"
	"time"

	"github.com/gofrs/uuid"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/drive/analytics/gotasks/models/cars"
	"a.yandex-team.ru/drive/analytics/gotasks/models/users"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/yt/go/mapreduce"
	"a.yandex-team.ru/yt/go/mapreduce/spec"
	"a.yandex-team.ru/yt/go/schema"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yson"
	"a.yandex-team.ru/zootopia/analytics/drive/models"
	"a.yandex-team.ru/zootopia/library/go/goyt"
)

func init() {
	updateExtendedCmd := cobra.Command{
		Use: "update-extended",
		Run: gotasks.WrapMain(updateExtendedMain),
	}
	updateExtendedCmd.Flags().String("yt-proxy", "hahn", "YT proxy")
	updateExtendedCmd.Flags().Bool("imei-log-v2", false, "Use IMEI log v2")
	updateExtendedCmd.Flags().String("imei-log-path", "", "Path to file with dropped IMEI log")
	CarsCmd.AddCommand(&updateExtendedCmd)
	// Register MR-tasks.
	mapreduce.Register(&extendedCarMapper{})
	mapreduce.Register(&extendedCarReducer{})
	mapreduce.Register(documentMapper{})
	mapreduce.Register(documentReducer{})
}

type IMEILog struct {
	Events []IMEIEvent `json:"events"`
}

func (l *IMEILog) UnmarshalFile(path string) error {
	bytes, err := ioutil.ReadFile(path)
	if err != nil {
		return err
	}
	return json.Unmarshal(bytes, l)
}

type IMEIEvent struct {
	Time int64   `json:"time"`
	ID   string  `json:"id"`
	IMEI *uint64 `json:"imei"`
}

func updateExtendedMain(ctx *gotasks.Context) (errMain error) {
	ytProxy, err := ctx.Cmd.Flags().GetString("yt-proxy")
	if err != nil {
		return err
	}
	yc, ok := ctx.YTs[ytProxy]
	if !ok {
		return fmt.Errorf("invalid YT proxy %q", ytProxy)
	}
	imeiLogPath, err := ctx.Cmd.Flags().GetString("imei-log-path")
	if err != nil {
		return err
	}
	var imeiLog IMEILog
	if imeiLogPath != "" {
		if err := imeiLog.UnmarshalFile(imeiLogPath); err != nil {
			return err
		}
	}
	tx, err := yc.BeginTx(ctx.Context, nil)
	if err != nil {
		return err
	}
	defer func() {
		if errMain == nil {
			errMain = tx.Commit()
			return
		}
		_ = tx.Abort()
	}()
	mr := mapreduce.New(yc).WithTx(tx)
	documentTable, err := goyt.TempTable(ctx.Context, tx, "")
	if err != nil {
		return err
	}
	defer func() {
		if errMain == nil {
			errMain = tx.RemoveNode(ctx.Context, documentTable, nil)
		}
	}()
	documentOpSpec := spec.Spec{
		InputTablePaths: []ypath.YPath{
			ctx.Config.YTPaths.CarDocumentHistoryTable,
			ctx.Config.YTPaths.CarDocumentAssignmentHistoryTable,
		},
		OutputTablePaths: []ypath.YPath{documentTable},
		ReduceBy:         []string{"doc_id"},
		SortBy:           []string{"doc_id", "event_id"},
		Pool:             "carsharing",
	}
	documentOp, err := mr.MapReduce(
		documentMapper{}, documentReducer{},
		documentOpSpec.MapReduce(),
	)
	if err != nil {
		return err
	}
	if err := documentOp.Wait(); err != nil {
		return err
	}
	tables := []ypath.YPath{}
	var tableParts []int
	addPart := func(path ypath.Path, kind int) {
		if len(path.String()) == 0 {
			ctx.Logger.Warn(
				"Table part disabled",
				log.String("path", path.String()),
				log.Int("kind", kind),
			)
			return
		}
		tables = append(tables, path)
		tableParts = append(tableParts, kind)
	}
	addPart(ctx.Config.YTPaths.SimpleCarsTable, carPart)
	addPart(ctx.Config.YTPaths.CarInfoHistoryTable, carHistoryPart)
	addPart(documentTable, carDocumentPart)
	outputTable := ctx.Config.YTPaths.ExtendedCarsTable
	opSpec := spec.Spec{
		InputTablePaths: tables,
		OutputTablePaths: []ypath.YPath{
			outputTable.Rich().
				SetSchema(schema.MustInfer(cars.ExtendedCar{})),
		},
		ReduceBy: []string{"car_id"},
		SortBy:   []string{"car_id"},
		Pool:     "carsharing",
	}
	now := time.Now().Unix()
	op, err := mr.MapReduce(
		&extendedCarMapper{Parts: tableParts, NowTime: now},
		&extendedCarReducer{
			NowTime: now,
			IMEILog: imeiLog,
		},
		opSpec.MapReduce(),
	)
	if err != nil {
		return err
	}
	if err := op.Wait(); err != nil {
		return err
	}
	sortOpSpec := spec.Spec{
		InputTablePaths: []ypath.YPath{outputTable},
		OutputTablePath: outputTable,
		SortBy:          []string{"car_id"},
		Pool:            "carsharing",
	}
	sortOp, err := mr.Sort(sortOpSpec.Sort())
	if err != nil {
		return err
	}
	return sortOp.Wait()
}

const (
	carPart = iota
	carHistoryPart
	carDocumentPart
)

type carDocumentMapperRow struct {
	Type    int    `yson:"type"`
	DocID   string `yson:"doc_id"`
	EventID int64  `yson:"event_id"`
	// Row contains raw value.
	Row yson.RawValue `yson:"data"`
}

type extendedCarMapperRow struct {
	Type  int    `yson:"type"`
	CarID string `yson:"car_id"`
	// Row contains raw value.
	Row yson.RawValue `yson:"data"`
}

type carDocument struct {
	ObjectID uuid.UUID
	IMEIs    []imeiEvent
	Heads    []headEvent
}

type imeiEvent struct {
	Time    int64
	EventID int64
	IMEI    uint64
}

type headEvent struct {
	Time    int64
	EventID int64
	HeadID  string
}

type imeiEventSorter []imeiEvent

func (v imeiEventSorter) Len() int {
	return len(v)
}

func (v imeiEventSorter) Less(i, j int) bool {
	if v[i].Time != v[j].Time {
		return v[i].Time < v[j].Time
	}
	return v[i].EventID < v[j].EventID
}

func (v imeiEventSorter) Swap(i, j int) {
	v[i], v[j] = v[j], v[i]
}

type headEventSorter []headEvent

func (v headEventSorter) Len() int {
	return len(v)
}

func (v headEventSorter) Less(i, j int) bool {
	if v[i].Time != v[j].Time {
		return v[i].Time < v[j].Time
	}
	return v[i].EventID < v[j].EventID
}

func (v headEventSorter) Swap(i, j int) {
	v[i], v[j] = v[j], v[i]
}

type documentMapper struct{}

func (o documentMapper) Do(
	_ mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	for in.Next() {
		rawRow := carDocumentMapperRow{Type: in.TableIndex()}
		switch rawRow.Type {
		case 0: // CarDocumentEvent.
			var row models.CarDocumentEvent
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.DocID = row.ID.String()
			rawRow.EventID = row.HistoryEventID
		case 1: // CarDocumentAssignmentEvent.
			var row models.CarDocumentAssignmentEvent
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.DocID = row.DocumentID.String()
			rawRow.EventID = row.HistoryEventID
		}
		if err := out[0].Write(rawRow); err != nil {
			return err
		}
	}
	return nil
}

func (o documentMapper) InputTypes() []interface{} {
	return []interface{}{
		models.CarDocumentEvent{},
		models.CarDocumentAssignmentEvent{},
	}
}

func (o documentMapper) OutputTypes() []interface{} {
	return []interface{}{carDocumentMapperRow{}}
}

type documentReducer struct{}

func (o documentReducer) Do(
	_ mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	return mapreduce.GroupKeys(in, func(in mapreduce.Reader) error {
		return o.reduceGroup(in, out)
	})
}

func (o documentReducer) reduceGroup(
	in mapreduce.Reader, out []mapreduce.Writer,
) error {
	type documentEvent struct {
		DocID  uuid.UUID
		Time   int64
		Type   int
		IMEI   uint64
		HeadID string
	}
	documents := map[uuid.UUID]documentEvent{}
	var attachEvents []models.CarDocumentAssignmentEvent
	for in.Next() {
		var rawRow carDocumentMapperRow
		if err := in.Scan(&rawRow); err != nil {
			return err
		}
		switch rawRow.Type {
		case 0: // CarDocumentEvent.
			var row models.CarDocumentEvent
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			switch row.Type {
			case "car_hardware_vega":
				var vega struct {
					IMEI string `json:"imei"`
				}
				if err := json.Unmarshal([]byte(row.Blob), &vega); err != nil {
					return err
				}
				imei, err := strconv.ParseUint(vega.IMEI, 10, 64)
				if err != nil {
					return err
				}
				documents[row.ID] = documentEvent{
					DocID: row.ID,
					Time:  row.HistoryTimestamp,
					Type:  1,
					IMEI:  imei,
				}
			case "car_hardware_head":
				var head struct {
					HeadID string `json:"head_id"`
				}
				if err := json.Unmarshal([]byte(row.Blob), &head); err != nil {
					return err
				}
				documents[row.ID] = documentEvent{
					DocID:  row.ID,
					Time:   row.HistoryTimestamp,
					Type:   2,
					HeadID: head.HeadID,
				}
			}
		case 1: // CarDocumentAssignmentEvent.
			var row models.CarDocumentAssignmentEvent
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			attachEvents = append(attachEvents, row)
		}
	}
	carDocuments := map[uuid.UUID]*carDocument{}
	for _, event := range attachEvents {
		if event.HistoryAction != "add" {
			continue
		}
		document, ok := documents[event.DocumentID]
		if !ok {
			continue
		}
		carDoc, ok := carDocuments[event.CarID]
		if !ok {
			carDoc = &carDocument{
				ObjectID: event.CarID,
			}
			carDocuments[event.CarID] = carDoc
		}
		switch document.Type {
		case 1:
			carDoc.IMEIs = append(carDoc.IMEIs, imeiEvent{
				EventID: event.HistoryEventID,
				Time:    event.HistoryTimestamp,
				IMEI:    document.IMEI,
			})
		case 2:
			carDoc.Heads = append(carDoc.Heads, headEvent{
				EventID: event.HistoryEventID,
				Time:    event.HistoryTimestamp,
				HeadID:  document.HeadID,
			})
		}
	}
	for _, document := range carDocuments {
		if err := out[0].Write(*document); err != nil {
			return err
		}
	}
	return nil
}

func castToInt(v interface{}) int {
	switch t := v.(type) {
	case int:
		return t
	case int64:
		return int(t)
	default:
		return 0
	}
}

func castToInt64(v interface{}) int64 {
	switch t := v.(type) {
	case int64:
		return t
	case int:
		return int64(t)
	default:
		return 0
	}
}

func castToString(v interface{}) string {
	switch t := v.(type) {
	case string:
		return t
	case []byte:
		return string(t)
	default:
		return ""
	}
}

func (o documentReducer) InputTypes() []interface{} {
	return []interface{}{map[string]interface{}{}}
}

func (o documentReducer) OutputTypes() []interface{} {
	return []interface{}{carDocument{}}
}

type extendedCarMapper struct {
	Parts   []int
	NowTime int64
}

func (e *extendedCarMapper) Do(
	_ mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	for in.Next() {
		var rawRow extendedCarMapperRow
		rawRow.Type = e.Parts[in.TableIndex()]
		switch rawRow.Type {
		case carPart:
			var row cars.Car
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.CarID = row.CarID.String()
		case carHistoryPart:
			var row models.CarEvent
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.CarID = row.ID.String()
		case carDocumentPart:
			var row carDocument
			if err := in.Scan(&row); err != nil {
				return err
			}
			var err error
			if rawRow.Row, err = yson.Marshal(row); err != nil {
				return err
			}
			rawRow.CarID = row.ObjectID.String()
		}
		if err := out[0].Write(rawRow); err != nil {
			return err
		}
	}
	return nil
}

func (e extendedCarMapper) InputTypes() []interface{} {
	return []interface{}{users.User{}, carDocument{}}
}

func (e extendedCarMapper) OutputTypes() []interface{} {
	return []interface{}{extendedCarMapperRow{}}
}

type extendedCarReducer struct {
	NowTime int64
	// IMEILog contains only dropped events.
	IMEILog IMEILog
}

func (e extendedCarReducer) Do(
	_ mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	return mapreduce.GroupKeys(in, func(in mapreduce.Reader) error {
		return e.reduceGroup(in, out)
	})
}

func (e *extendedCarReducer) reduceGroup(
	in mapreduce.Reader, out []mapreduce.Writer,
) error {
	droppedIMEILog := map[string][]imeiEvent{}
	for _, event := range e.IMEILog.Events {
		if event.IMEI != nil {
			droppedIMEILog[event.ID] = append(droppedIMEILog[event.ID], imeiEvent{
				Time: event.Time,
				IMEI: *event.IMEI,
			})
		}
	}
	var car cars.ExtendedCar
	var carHeads []headEvent
	var imeiHistory []imeiEvent
	for in.Next() {
		var rawRow extendedCarMapperRow
		if err := in.Scan(&rawRow); err != nil {
			return err
		}
		switch rawRow.Type {
		case carPart:
			var row cars.Car
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			car.Car = row
		case carHistoryPart:
			var row models.CarEvent
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			imeiHistory = append(imeiHistory, imeiEvent{
				Time:    row.HistoryTimestamp,
				EventID: row.HistoryEventID,
				IMEI:    uint64(row.IMEI),
			})
		case carDocumentPart:
			var row carDocument
			if err := yson.Unmarshal(rawRow.Row, &row); err != nil {
				return err
			}
			carHeads = append(carHeads, row.Heads...)
		}
	}
	if car.CarID == uuid.Nil {
		return nil
	}
	if history, ok := droppedIMEILog[car.CarID.String()]; ok {
		imeiHistory = append(imeiHistory, history...)
	}
	sort.Sort(imeiEventSorter(imeiHistory))
	if len(imeiHistory) > 0 {
		prev := imeiHistory[0]
		for _, event := range imeiHistory {
			if prev.IMEI != event.IMEI {
				if prev.IMEI != 0 {
					car.IMEILog = append(car.IMEILog, cars.IMEI{
						BeginTime: prev.Time,
						EndTime:   event.Time,
						IMEI:      prev.IMEI,
					})
				}
				prev = event
			}
		}
		if prev.IMEI != 0 {
			car.IMEILog = append(car.IMEILog, cars.IMEI{
				BeginTime: prev.Time,
				IMEI:      prev.IMEI,
			})
		}
		if car.IMEI != 0 {
			if len(car.IMEILog) > 0 {
				last := car.IMEILog[len(car.IMEILog)-1]
				if last.IMEI != car.IMEI {
					if last.EndTime == 0 {
						return fmt.Errorf(
							"inconsistent imei for car %q: %d != %d",
							car.CarID, last.IMEI, car.IMEI,
						)
					}
					car.IMEILog = append(car.IMEILog, cars.IMEI{
						BeginTime: last.EndTime,
						IMEI:      car.IMEI,
					})
				}
			} else {
				car.IMEILog = append(car.IMEILog, cars.IMEI{
					BeginTime: 0,
					IMEI:      car.IMEI,
				})
			}
		}
	}
	if car.IMEI == 0 && len(car.IMEILog) > 0 {
		car.IMEI = car.IMEILog[len(car.IMEILog)-1].IMEI
	}
	sort.Sort(headEventSorter(carHeads))
	if len(carHeads) > 0 {
		prev := carHeads[0]
		for _, event := range carHeads {
			if prev.HeadID != event.HeadID {
				car.HeadLog = append(car.HeadLog, cars.Head{
					BeginTime: prev.Time,
					EndTime:   event.Time,
					HeadID:    prev.HeadID,
				})
				prev = event
			}
		}
		car.HeadLog = append(car.HeadLog, cars.Head{
			BeginTime: prev.Time,
			HeadID:    prev.HeadID,
		})
	}
	if len(car.HeadLog) > 0 {
		car.HeadID = car.HeadLog[len(car.HeadLog)-1].HeadID
	}
	return out[0].Write(car)
}

func (e extendedCarReducer) InputTypes() []interface{} {
	return []interface{}{extendedCarMapperRow{}}
}

func (e extendedCarReducer) OutputTypes() []interface{} {
	return []interface{}{cars.ExtendedCar{}}
}
