package gotasks

import (
	"context"
	"fmt"
	"os"
	"os/signal"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gobase/config"
	"a.yandex-team.ru/drive/analytics/gobase/core"
	"a.yandex-team.ru/drive/library/go/gocfg"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/yt/go/yt"
)

// Context represents command context.
type Context struct {
	*core.Core
	// Cmd contains cobra.Command.
	Cmd *cobra.Command
	// Args contains command args.
	Args []string
	// Logger contains logger.
	Logger log.Logger
	// Context contains context.
	Context context.Context
}

// GetYT returns yt proxy.
func (c *Context) GetYT() (yt.Client, error) {
	proxy, err := c.Cmd.Flags().GetString("yt-proxy")
	if err != nil {
		return nil, err
	}
	client, ok := c.YTs[proxy]
	if !ok {
		return nil, fmt.Errorf("proxy %q does not exists", proxy)
	}
	return client, nil
}

var (
	// RootCmd represents console commands root.
	RootCmd cobra.Command
)

func init() {
	RootCmd.PersistentFlags().String("config", "config.json", "Path to config file")
	RootCmd.PersistentFlags().String("log-level", "", "Rewrite config log level")
}

// WrapMain wraps main function for command.
func WrapMain(
	main func(ctx *Context) error,
) func(cmd *cobra.Command, args []string) {
	return func(cmd *cobra.Command, args []string) {
		path, err := cmd.Flags().GetString("config")
		if err != nil {
			panic(err)
		}
		cfg := config.Config{
			LogLevel: log.InfoLevel,
		}
		if err := gocfg.ParseFile(path, &cfg); err != nil {
			panic(err)
		}
		logLevel, err := cmd.Flags().GetString("log-level")
		if err != nil {
			panic(err)
		}
		if logLevel != "" {
			level, err := log.ParseLevel(logLevel)
			if err != nil {
				panic(fmt.Errorf("unable to parse log level: %w", err))
			}
			cfg.LogLevel = level
		}
		logger, err := zap.New(zap.ConsoleConfig(cfg.LogLevel))
		if err != nil {
			panic(err)
		}
		cr, err := core.NewCore(&cfg, core.WithLogger(logger))
		if err != nil {
			panic(err)
		}
		ctx := Context{
			Core:   cr,
			Cmd:    cmd,
			Args:   args,
			Logger: logger,
		}
		if err := cr.Start(); err != nil {
			panic(err)
		}
		defer cr.Stop()
		var cancel context.CancelFunc
		ctx.Context, cancel = context.WithCancel(context.Background())
		exit := make(chan os.Signal, 1)
		signal.Notify(exit, os.Interrupt)
		defer func() {
			signal.Stop(exit)
			cancel()
		}()
		go func() {
			select {
			case <-ctx.Context.Done():
			case <-exit:
				ctx.Logger.Warn("Canceling execution...")
				// This allows to force cancelng by twice press on Ctrl+C.
				signal.Stop(exit)
				cancel()
			}
		}()
		if err := main(&ctx); err != nil {
			panic(err)
		}
	}
}
