package data

import (
	"fmt"
	"strings"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/yt/go/mapreduce"
	"a.yandex-team.ru/yt/go/mapreduce/spec"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
)

func init() {
	ytCopyCmd := cobra.Command{
		Use: "yt-copy",
		Run: gotasks.WrapMain(ytCopyMain),
	}
	ytCopyCmd.Flags().String("from", "", "proxy:path")
	ytCopyCmd.Flags().String("to", "", "proxy:path")
	ytCopyCmd.Flags().String("optimize-for", "", "")
	DataCmd.AddCommand(&ytCopyCmd)
}

func createTable(
	ctx *gotasks.Context, tx yt.CypressClient, path ypath.YPath,
) error {
	if ok, err := tx.NodeExists(ctx.Context, path, nil); err != nil {
		return err
	} else if ok {
		if err := tx.RemoveNode(ctx.Context, path, nil); err != nil {
			return err
		}
	}
	_, err := tx.CreateNode(ctx.Context, path, yt.NodeTable, nil)
	return err
}

func ytCopyMain(ctx *gotasks.Context) (errResult error) {
	from, _ := ctx.Cmd.Flags().GetString("from")
	to, _ := ctx.Cmd.Flags().GetString("to")
	optimizeFor, _ := ctx.Cmd.Flags().GetString("optimize-for")
	fromParts := strings.SplitN(from, ":", 2)
	toParts := strings.SplitN(to, ":", 2)
	if len(fromParts) != 2 {
		return fmt.Errorf("invalid '--from' value")
	}
	if len(toParts) != 2 {
		return fmt.Errorf("invalid '--to' value")
	}
	yc, ok := ctx.YTs[toParts[0]]
	if !ok {
		return fmt.Errorf("invalid YT proxy %q", toParts[0])
	}
	if _, ok := ctx.YTs[fromParts[0]]; !ok {
		return fmt.Errorf("invalid YT proxy %q", fromParts[0])
	}
	tx, err := yc.BeginTx(ctx.Context, nil)
	if err != nil {
		return err
	}
	defer func() {
		if errResult != nil {
			_ = tx.Abort()
			return
		}
		errResult = tx.Commit()
	}()
	toPath := ypath.Path(toParts[1])
	if err := createTable(ctx, tx, toPath); err != nil {
		return err
	}
	mr := mapreduce.New(yc).WithTx(tx)
	opSpec := spec.Spec{
		Type:            yt.OperationRemoteCopy,
		InputTablePaths: []ypath.YPath{ypath.Path(fromParts[1])},
		OutputTablePath: toPath,
		ClusterName:     fromParts[0],
	}
	op, err := mr.RemoteCopy(&opSpec)
	if err != nil {
		return err
	}
	if err := op.Wait(); err != nil {
		return err
	}
	if len(optimizeFor) > 0 {
		if err := tx.SetNode(ctx.Context, toPath.Attr("optimize_for"), optimizeFor, nil); err != nil {
			return err
		}
		opSpec := spec.Spec{
			InputTablePaths: []ypath.YPath{toPath},
			OutputTablePath: toPath,
			ForceTransform:  true,
			MergeMode:       "ordered",
		}
		op, err := mr.Merge(opSpec.Merge())
		if err != nil {
			return err
		}
		if err := op.Wait(); err != nil {
			return err
		}
	}
	return nil
}
