package exports

import (
	"fmt"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/analytics/gotasks"
	"a.yandex-team.ru/yt/go/mapreduce"
	"a.yandex-team.ru/zootopia/analytics/drive/models"
)

func init() {
	updateCarTagsHistoryCmd := cobra.Command{
		Use: "update-car-tags-history",
		Run: gotasks.WrapMain(updateCarTagsHistoryMain),
	}
	updateCarTagsHistoryCmd.Flags().String("yt-proxy", "hahn", "")
	ExportsCmd.AddCommand(&updateCarTagsHistoryCmd)
}

func updateCarTagsHistoryMain(ctx *gotasks.Context) error {
	yc, err := ctx.GetYT()
	if err != nil {
		return err
	}
	db, ok := ctx.DBs[ctx.Config.BackendDB]
	if !ok {
		return fmt.Errorf("database %q not found", ctx.Config.BackendDB)
	}
	exporter := NewExporter(
		models.CarTagEvent{}, "history_event_id",
		yc, ctx.Config.YTPaths.CarTagsHistoryTable,
		db, "car_tags_history",
		ctx.Config.YTPaths.BackendCarTagsHistoryDir,
		carTagsHistoryMapper{},
	)
	return exporter.Export(ctx)
}

type rawCarTagEvent struct {
	HistoryEventID      interface{} `yson:"history_event_id"`
	HistoryAction       string      `yson:"history_action"`
	HistoryTimestamp    interface{} `yson:"history_timestamp"`
	HistoryUserID       string      `yson:"history_user_id"`
	HistoryOriginatorID interface{} `yson:"history_originator_id"`
	HistoryComment      interface{} `yson:"history_comment"`
	ID                  interface{} `yson:"tag_id"`
	ObjectID            interface{} `yson:"object_id"`
	Tag                 interface{} `yson:"tag"`
	Data                string      `yson:"data"`
	Performer           interface{} `yson:"performer"`
	Priority            interface{} `yson:"priority"`
	Snapshot            interface{} `yson:"snapshot"`
}

func (e rawCarTagEvent) Parse() (event models.CarTagEvent, err error) {
	event.HistoryEventID = castToInt64(e.HistoryEventID)
	event.HistoryAction = e.HistoryAction
	event.HistoryTimestamp = castToInt64(e.HistoryTimestamp)
	event.HistoryUserID = e.HistoryUserID
	event.HistoryOriginatorID = castToNString(e.HistoryOriginatorID)
	event.HistoryComment = castToNString(e.HistoryComment)
	event.ID = castToUUID(e.ID)
	event.ObjectID = castToUUID(e.ObjectID)
	event.Tag = castToNString(e.Tag)
	event.Data = e.Data
	event.Performer = castToNString(e.Performer)
	event.Priority = castToNInt64(e.Priority)
	event.Snapshot = castToNString(e.Snapshot)
	return
}

type carTagsHistoryMapper struct{}

func (carTagsHistoryMapper) InputTypes() []interface{} {
	return []interface{}{rawCarTagEvent{}}
}

func (carTagsHistoryMapper) OutputTypes() []interface{} {
	return []interface{}{models.CarTagEvent{}}
}

func (carTagsHistoryMapper) Do(
	ctx mapreduce.JobContext, in mapreduce.Reader, out []mapreduce.Writer,
) error {
	for in.Next() {
		var row rawCarTagEvent
		if err := in.Scan(&row); err != nil {
			return err
		}
		event, err := row.Parse()
		if err != nil {
			return err
		}
		if err := out[0].Write(event); err != nil {
			return err
		}
	}
	return nil
}
