package exports

import (
	"context"
	"encoding/json"
	"log"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/drive/analytics/gotasks/models/cars"
	"a.yandex-team.ru/yt/go/schema"
	"a.yandex-team.ru/yt/go/yt"
)

func carsExportProcess(service *Service, closer <-chan struct{}) {
	e := carsExport{service, closer}
	e.Run()
}

type carsExport struct {
	service *Service
	closer  <-chan struct{}
}

func (e *carsExport) Run() {
	ticker := time.NewTicker(30 * time.Minute)
	defer ticker.Stop()
	log.Println("Cars export running...")
	if err := e.run(); err != nil {
		log.Println("Error:", err)
	}
	log.Println("Cars export finished...")
	for {
		select {
		case <-e.closer:
			log.Println("Cars export exiting...")
			return
		case <-ticker.C:
			log.Println("Cars export running...")
			if err := e.run(); err != nil {
				log.Println("Error:", err)
			}
			log.Println("Cars export finished...")
		}
	}
}

func (e *carsExport) run() error {
	// Setup Yt client.
	yc := e.service.YT
	// Create table if does not exists.
	tableSchema, err := schema.Infer(cars.Car{})
	if err != nil {
		return err
	}
	tablePath := e.service.Config.YTPaths.SimpleCarsTable.Rich().
		SetSchema(tableSchema)
	if _, err := yc.CreateNode(
		context.TODO(), tablePath, yt.NodeTable,
		&yt.CreateNodeOptions{IgnoreExisting: true},
	); err != nil {
		return err
	}
	assignmentIDs := e.service.CarDocumentAssignments.GetIDs()
	carsByID := map[uuid.UUID]*cars.Car{}
	allCars, err := e.service.Cars.All()
	if err != nil {
		return err
	}
	for _, car := range allCars {
		carsByID[car.ID] = &cars.Car{
			CarID:           car.ID,
			VIN:             string(car.VIN),
			IMEI:            uint64(car.IMEI),
			Model:           string(car.ModelCode),
			Tags:            []string{},
			UnsafeDocuments: map[string][]interface{}{},
		}
	}
	for _, id := range assignmentIDs {
		assignment, ok := e.service.CarDocumentAssignments.Get(id)
		if !ok {
			continue
		}
		if document, ok := e.service.CarDocuments.Get(
			assignment.DocumentID,
		); ok {
			var unsafeDocument interface{}
			if err := json.Unmarshal(
				[]byte(document.Blob), &unsafeDocument,
			); err != nil {
				continue
			}
			car, ok := carsByID[assignment.CarID]
			if !ok {
				continue
			}
			car.UnsafeDocuments[document.Type] = append(
				car.UnsafeDocuments[document.Type], unsafeDocument,
			)
		}
	}
	for _, id := range e.service.CarTags.GetIDs() {
		tag, ok := e.service.CarTags.Get(id)
		if !ok {
			continue
		}
		car, ok := carsByID[tag.TargetID]
		if !ok {
			continue
		}
		car.Tags = append(car.Tags, string(tag.Tag))
	}
	// Create writer to YT
	writer, err := yc.WriteTable(context.TODO(), tablePath, nil)
	if err != nil {
		return err
	}
	// Iterate over IDs
	for _, car := range carsByID {
		if err := writer.Write(*car); err != nil {
			return err
		}
	}
	// Apply updates
	return writer.Commit()
}
